package ru.yandex.direct.mysql.ytsync.task.config;

import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableSet;
import com.typesafe.config.Config;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.mysql.ytsync.synchronizator.util.SnapshotConfig;
import ru.yandex.direct.mysql.ytsync.synchronizator.util.SyncConfig;
import ru.yandex.direct.ytwrapper.model.YtCluster;

public class SyncConfigImpl extends RootPathConfig implements SyncConfig {
    private final Config primarySettings;
    private final Config pathSettings;
    private final String clusterName;
    private final Config perClusterSettings;
    private final int mysqlServerId;
    private final List<String> dbNames;

    public SyncConfigImpl(DirectConfig directConfig, List<String> dbNames) {
        super(directConfig);

        this.primarySettings = directConfig.getBranch("yt-sync").getConfig();
        this.dbNames = dbNames;
        this.pathSettings = primarySettings.getConfig("path");

        // Следующие 2 опции получаем сразу при инициализации с целью упасть, если их не окажется
        this.clusterName = primarySettings.getString("cluster");
        this.mysqlServerId = primarySettings.getInt("mysql-server-id");

        this.perClusterSettings = createPerClusterSettings();
    }

    private Config createPerClusterSettings() {
        Config settings = primarySettings.getConfig("default");
        if (primarySettings.hasPath(clusterName)) {
            // Настройки кластера переопределяют настройки из default
            settings = primarySettings.getConfig(clusterName).withFallback(settings);
        }
        if (primarySettings.hasPath("all")) {
            // Настройки all переопределяют настройки кластера
            settings = primarySettings.getConfig("all").withFallback(settings);
        }
        return settings;
    }

    /**
     * Список баз для синхронизации
     * Если не задано в конфиге, получаем на основе данных о количестве шардов
     */
    @Override
    public List<String> getDbNames() {
        return dbNames;
    }

    @Override
    public YtCluster cluster() {
        return YtCluster.parse(clusterName);
    }

    /**
     * Импортировать все таблицы из mysql
     */
    @Override
    public boolean importAllTables() {
        return perClusterSettings.getBoolean("import-all-tables");
    }

    /**
     * Множество таблиц, исключаемых из синхронизации
     * Допускается использование регулярных выражений
     */
    @Override
    public Set<String> getExcludeTableNames() {
        return ImmutableSet.copyOf(perClusterSettings.getStringList("exclude-table-names").stream()
                .map(String::toLowerCase)
                .collect(Collectors.toSet()));
    }

    /**
     * Идентификатор сервера для использования при чтении binlog'ов
     */
    @Override
    public int mysqlServerId() {
        return mysqlServerId;
    }

    /**
     * Разрешено ли использовать большие транзакции при использовании rpc клиента
     */
    @Override
    public boolean useRpcTransactions() {
        return perClusterSettings.getBoolean("use-rpc-transactions");
    }

    /**
     * Путь ссылки указывающую на папку из которой grid-core читает данные из YT'а
     */
    @Override
    public String stableLinkPath() {
        return pathSettings.getString("stable-link");
    }

    @Override
    public SnapshotConfig snapshotConfig() {
        Config snapshotCfg = perClusterSettings.getConfig("daily-snapshots");
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("HH:mm:ss");
        boolean enabled = snapshotCfg.getBoolean("enabled");
        LocalTime intervalFrom = LocalTime.parse(snapshotCfg.getString("interval-from"), formatter);
        LocalTime intervalTo = LocalTime.parse(snapshotCfg.getString("interval-to"), formatter);
        boolean enabledOnPrestable = snapshotCfg.getBoolean("enabled-on-prestable");
        return new SnapshotConfig(enabled, intervalFrom, intervalTo, enabledOnPrestable);
    }

    /**
     * Префикс для путей с локами синхронизации баз
     */
    @Override
    public String syncLocksPrefix() {
        return rootPath() + pathSettings.getString("sync-locks-prefix");
    }

    /**
     * Максимальное кол-во операций, которые желательно буферизировать перед записью в yt
     */
    @Override
    public int maxPendingOperations() {
        return perClusterSettings.getInt("max-pending-operations");
    }

    /**
     * Период в миллисекундах с которым агрегатор сбрасывает данные в yt
     */
    @Override
    public int flushEveryMillis() {
        return perClusterSettings.getInt("flush-every-ms");
    }

    /**
     * Если в течение значительного времени при попытках записи в YT мы не получаем ничего, кроме ошибок,
     * то имеет смысл завершить поток (а следом и весь процесс, и дать шанс другому инстансу)
     */
    @Override
    public int globalProgressTimeoutSeconds() {
        return perClusterSettings.getInt("global-progress-timeout-seconds");
    }

    /**
     * GtidSet по достижении которого нужно прекратить чтение binlog'а из базы dbName
     */
    @Override
    public String streamStopAfterSet(String dbName) {
        String path = "stop-after.\"" + dbName + "\"";
        if (perClusterSettings.hasPath(path)) {
            String gtidSet = perClusterSettings.getString(path);
            if (!gtidSet.isEmpty()) {
                return gtidSet;
            }
        }
        return null;
    }

    @Override
    public boolean verifyChecksums() {
        return perClusterSettings.getBoolean("verify-checksums");
    }

    @Override
    public boolean processReshardingEvents() { return perClusterSettings.getBoolean("process-resharding-events"); }
}
