package ru.yandex.direct.mysql.ytsync.task.provider;

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import one.util.streamex.StreamEx;

import ru.yandex.direct.mysql.MySQLSimpleRow;
import ru.yandex.direct.mysql.ytsync.common.compatibility.YtSupport;
import ru.yandex.direct.mysql.ytsync.synchronizator.tableprocessors.SyncFlatRowCreatorBase;
import ru.yandex.direct.mysql.ytsync.synchronizator.tableprocessors.SyncFlatRowCreatorPack;
import ru.yandex.direct.mysql.ytsync.synchronizator.tableprocessors.TableProcessor;
import ru.yandex.direct.mysql.ytsync.synchronizator.tableprocessors.TaskBasedTableProcessor;
import ru.yandex.direct.mysql.ytsync.task.config.DirectYtSyncConfig;

import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@ParametersAreNonnullByDefault
public class TaskProviderUtil {
    private TaskProviderUtil() {
    }

    public static List<TableProcessor> createTableProcessors(List<TaskProvider> taskProviders, YtSupport yt,
                                                             DirectYtSyncConfig config) {
        return StreamEx.of(taskProviders)
                .mapToEntry(p -> p.getSyncTask().getTablePath(config), Function.identity())
                .sorted(Comparator.comparing(Map.Entry::getKey))
                .collapseKeys()
                .mapValues(ps -> createTableProcessor(ps, yt, config))
                .values()
                .toList();
    }

    private static TableProcessor createTableProcessor(List<TaskProvider> taskProviders, YtSupport yt,
                                                       DirectYtSyncConfig config) {
        if (taskProviders.size() == 1) {
            TaskProvider task = taskProviders.get(0);
            return createSimpleTableProcessor(yt, config, task);
        }
        TaskProvider mainTask = taskProviders.get(0);
        Set<String> allTables = taskProviders.stream()
                .map(p -> p.getSyncTask().getAllTables())
                .flatMap(Collection::stream)
                .collect(Collectors.toSet());

        Map<String, Predicate<MySQLSimpleRow>> skipPredicates = taskProviders.stream()
                .filter(TaskProvider::hasSkipPredicate)
                .collect(Collectors.toMap(TaskProvider::getMainTable, TaskProvider::getSkipPredicate));
        return new TaskBasedTableProcessor(
                ImmutableSet.of(config.getBaseSchemaName(), config.getPpcDictSchemaName()),
                listToSet(taskProviders, TaskProvider::getMainTable),
                skipPredicates,
                allTables,
                mergeCreators(taskProviders),
                mainTask.getSyncTask().getIndexSchema(),
                mainTask.constructTable(yt, config)
        );
    }

    /**
     * Создает {@link TableProcessor} с одной исходной таблицей
     */
    public static TableProcessor createSimpleTableProcessor(YtSupport yt, DirectYtSyncConfig config, TaskProvider task) {
        return new TaskBasedTableProcessor(
                ImmutableSet.of(config.getBaseSchemaName(), config.getPpcDictSchemaName()),
                Collections.singleton(task.getMainTable()),
                task.getSyncTask().getAllTables(),
                task.getSyncTask().getSyncFlatRowCreator(),
                task.getSyncTask().getIndexSchema(),
                task.constructTable(yt, config)
        );
    }

    private static SyncFlatRowCreatorBase mergeCreators(List<TaskProvider> taskProviders) {
        Map<String, SyncFlatRowCreatorBase> creators = taskProviders.stream()
                .collect(Collectors.toMap(TaskProvider::getMainTable, tp -> tp.getSyncTask().getSyncFlatRowCreator()));
        return new SyncFlatRowCreatorPack(creators);
    }
}
