package ru.yandex.direct.mysql.ytsync.util;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.time.Duration;
import java.util.regex.Pattern;

import com.google.common.collect.Lists;

import ru.yandex.direct.db.config.DbConfig;
import ru.yandex.direct.mysql.MySQLUtils;
import ru.yandex.direct.mysql.ytsync.task.config.DirectYtSyncConfig;

public class MySqlUtils {
    private static final Duration CONNECT_TIMEOUT = Duration.ofSeconds(10);

    public static Connection mysqlConnect(DbConfig dbConfig) throws SQLException {
        return MySQLUtils.connect(
                dbConfig.getHosts().get(0),
                dbConfig.getPort(),
                dbConfig.getUser(),
                dbConfig.getPass(),
                CONNECT_TIMEOUT);
    }

    /**
     * Проверяет, является ли хост mysqld из MDB
    */
    public static boolean isMdbHost(String hostname) {
        return hostname.toLowerCase().endsWith(".db.yandex.net");
    }

    /**
     * Возвращает автоматически выбранный slave хост для указанного dbConfig мастера
     */
    public static DbConfig selectSlaveAuto(DirectYtSyncConfig config, DbConfig dbConfig) throws SQLException {
        Pattern pattern = config.slaveHostPattern();
        try (Connection connection = mysqlConnect(dbConfig)) {
            try (PreparedStatement stmt = connection.prepareStatement("SHOW SLAVE HOSTS")) {
                try (ResultSet rs = stmt.executeQuery()) {
                    while (rs.next()) {
                        String slaveHost = rs.getString("Host");
                        if (pattern.matcher(slaveHost).matches()) {
                            int slavePort = rs.getInt("Port");
                            if (isMdbHost(slaveHost)) {
                                slavePort = 3306;
                            }
                            DbConfig slaveConfig = new DbConfig();
                            slaveConfig.setDbName(dbConfig.getDbName());
                            slaveConfig.setEngine(dbConfig.getEngine());
                            slaveConfig.setHosts(Lists.newArrayList(slaveHost));
                            slaveConfig.setPort(slavePort);
                            slaveConfig.setDb(dbConfig.getDb());
                            slaveConfig.setUser(dbConfig.getUser());
                            slaveConfig.setPass(dbConfig.getPass());
                            return slaveConfig;
                        }
                    }
                }
            }
        }
        throw new IllegalArgumentException(
                "Cannot choose slave for " + dbConfig.getDbName() + ": current master at " + dbConfig.getHosts().get(0)
                        + " has no matching slaves");
    }
}
