package ru.yandex.direct.useractionlog;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * mysql server uuid и event id
 */
@ParametersAreNonnullByDefault
public class Gtid implements Map.Entry<String, Long> {
    private final String uuid;
    private final long eventId;

    public Gtid(String uuid, long eventId) {
        this.uuid = uuid;
        this.eventId = eventId;
    }

    public Gtid(Map.Entry<String, Long> entry) {
        this(entry.getKey(), entry.getValue());
    }

    /**
     * Парсинг строки {@literal <server uuid>:<event id>}
     */
    public static Gtid fromGtidString(String gtid) {
        checkServerUuid(gtid);
        return new Gtid(gtid.substring(0, 36), Long.parseLong(gtid.substring(37)));
    }

    /**
     * Парсинг строки {@literal <server uuid>:<start event id>-<end event id>}
     * {@link #getEventId()} будет выдавать {@literal <end event id>},
     * а {@literal <start event id>} будет проигнорирован.
     */
    public static Gtid fromGtidIntervalString(String gtidInterval) {
        checkServerUuid(gtidInterval);
        String[] interval = gtidInterval.substring(37).split("-");
        String serverUuid = gtidInterval.substring(0, 36);
        switch (interval.length) {
            case 1:
                return new Gtid(serverUuid, Long.parseLong(interval[0]));
            case 2:
                return new Gtid(serverUuid, Long.parseLong(interval[1]));
            default:
                throw new IllegalArgumentException(gtidInterval + " is not a valid mysql gtid interval");
        }
    }

    /**
     * Парсинг строки {@literal <server uuid>:<start event id>-<end event id>,...}
     * В каждом результирующем {@link Gtid} {@link #getEventId()} будет выдавать соответствующий
     * {@literal <end event id>}, а все {@literal <start event id>} будут проигнорированы.
     */
    public static List<Gtid> fromGtidSet(String gtidSet) {
        List<Gtid> result = new ArrayList<>();
        for (String gtidInterval : gtidSet.split(",")) {
            result.add(fromGtidIntervalString(gtidInterval.trim()));
        }
        return result;
    }

    private static void checkServerUuid(String gtid) {
        if (gtid.length() < 37 || gtid.charAt(36) != ':') {
            throw new IllegalArgumentException(gtid + " is not a valid mysql gtid");
        }
    }

    public String getUuid() {
        return uuid;
    }

    public long getEventId() {
        return eventId;
    }

    @Override
    public String toString() {
        return uuid + ':' + eventId;
    }

    @Override
    public String getKey() {
        return getUuid();
    }

    @Override
    public Long getValue() {
        return getEventId();
    }

    @Override
    public Long setValue(Long value) {
        throw new UnsupportedOperationException("unmodifiable object");
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        Gtid gtid = (Gtid) o;
        return eventId == gtid.eventId &&
                Objects.equals(uuid, gtid.uuid);
    }

    @Override
    public int hashCode() {
        return Objects.hash(uuid, eventId);
    }
}
