package ru.yandex.direct.useractionlog;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.dbutil.wrapper.DatabaseWrapper;
import ru.yandex.direct.mysql.MySQLBinlogState;
import ru.yandex.direct.useractionlog.db.ReadActionLogTable;
import ru.yandex.direct.useractionlog.db.ReadStateTable;
import ru.yandex.direct.useractionlog.db.StateReader;

@ParametersAreNonnullByDefault
public class LatestRecordTimeFetcher {
    private final StateReader stateReader;
    private final ReadActionLogTable readActionLogTable;

    public LatestRecordTimeFetcher(Function<String, DatabaseWrapper> fn) {
        this.stateReader = new ReadStateTable(fn, TableNames.USER_ACTION_LOG_STATE_TABLE);
        this.readActionLogTable = new ReadActionLogTable(fn, TableNames.READ_USER_ACTION_LOG_TABLE);
    }

    /**
     * Получить для каждого источника время последней записи. Для этого из {@link ReadStateTable}
     * забирается последний state для каждого источника, из него берётся список uuid-ов.
     * Для каждого uuid-а достаётся время последней записи из {@link ReadActionLogTable}, из
     * них выбирается максимальное время для каждого источника. Если ни одной записи не нашлось,
     * возвращается {@link LocalDateTime#MIN}
     *
     * @param minDate минимальная рассматриваемая дата
     */
    public Map<String, LocalDateTime> getLatestRecordTime(LocalDate minDate) {
        Map<String, MySQLBinlogState> allLogStates = stateReader.getAllActualLogStates();

        Map<String, Collection<String>> sourceUuids = new HashMap<>();
        Collection<String> allUuids = new ArrayList<>();
        allLogStates.forEach((source, state) -> {
            if (state != null) {
                for (String gtid : state.getGtidSet().split(",")) {
                    String uuid = gtid.split(":")[0];
                    sourceUuids.computeIfAbsent(source, ignored -> new ArrayList<>())
                            .add(uuid);
                    allUuids.add(uuid);
                }
            }
        });
        Map<String, LocalDateTime> dateTimeForUuid = readActionLogTable.getLatestRecordTimeForUuids(
                minDate, allUuids);
        Map<String, LocalDateTime> result = new HashMap<>();
        for (Map.Entry<String, Collection<String>> entry : sourceUuids.entrySet()) {
            LocalDateTime maxDateTime = LocalDateTime.MIN;
            for (String uuid : entry.getValue()) {
                LocalDateTime uuidDateTime = dateTimeForUuid.getOrDefault(uuid, LocalDateTime.MIN);
                if (uuidDateTime.isAfter(maxDateTime)) {
                    maxDateTime = uuidDateTime;
                }
            }
            result.put(entry.getKey(), maxDateTime);
        }
        return result;
    }
}
