package ru.yandex.direct.useractionlog.db;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLRecoverableException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicIntegerArray;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.ThreadSafe;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.apache.logging.log4j.util.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.clickhouse.ClickHouseDataSource;
import ru.yandex.clickhouse.except.ClickHouseErrorCode;
import ru.yandex.clickhouse.except.ClickHouseException;
import ru.yandex.direct.db.config.DbConfig;
import ru.yandex.direct.db.config.DbConfigException;
import ru.yandex.direct.db.config.DbConfigFactory;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapper;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapperProvider;
import ru.yandex.direct.useractionlog.TableNames;
import ru.yandex.direct.utils.Completer;
import ru.yandex.direct.utils.Interrupts;

/**
 * Менеджер соединений с кластером ClickHouse, который берёт информацию о кластере из DbConfig.
 * <p>
 * Все MergeTree-таблицы можно разбить на два класса:
 * <ol>
 * <li>
 * Таблицы из M шардов, в каждом N реплик. Такими таблицами являются: <ul>
 * <li>{@link ru.yandex.direct.useractionlog.schema.ActionLogSchema}</li>
 * <li>{@link ru.yandex.direct.useractionlog.schema.dict.DictSchema}</li>
 * </ul>
 * </li>
 * <li>Таблицы из одного шарда, в котором N реплик. Для этих таблиц важна
 * линеаризуемость операций вставки и чтения. Такими таблицами являются: <ul>
 * <li>{@link ru.yandex.direct.useractionlog.schema.dict.DictSchema}</li>
 * <li>{@link ru.yandex.direct.useractionlog.schema.StateSchema}</li>
 * </ul>
 * Позволяется указать несколько шардов для таких таблиц, но тогда все шарды, кроме шарда с минимальным номером, не будут использоваться.
 * </li>
 * </ol>
 * <p>
 * Для чтения MxN таблиц должна существовать Distributed-таблица, которая агрегирует данные из каждого
 * ReplicatedMergeTree-шарда. Для чтения и для записи может быть выбрана любая реплика любого шарда
 * (реплики и шарды выбираются по round-robin).
 * <p>
 * <p>
 * Для 1xN таблиц нет никакой балансировки. При инициализации объекта выбирается один общий хост.
 * Все операции чтения и записи будут отправляться на этот хост.
 * <p>
 * dbconfig должен быть в таком же формате, как генерирует {@code ClickHouseClusterDbConfig}.
 */
// TODO(lagunov) слишком сложный класс. Нужно попытаться не зависеть от двух интерфейсов.
@ParametersAreNonnullByDefault
@ThreadSafe
public class DbConfigClickHouseManager implements ShardReplicaChooser, ClickHouseClusterInfo {
    private static final Set<ClusterDbLabel> NEED_LINEARIZABLE_INSERTS = EnumSet.of(
            ClusterDbLabel.DICT,
            ClusterDbLabel.STATE);
    private static final Logger logger = LoggerFactory.getLogger(DbConfigClickHouseManager.class);

    private final DatabaseWrapperProvider databaseWrapperProvider;

    /**
     * clickhouse, в который будет ходить приложение для записи/чтения таблиц, в которых важна линеаризуемость
     * вставки/чтения
     */
    private final String replicaPathForAllSingleShard;

    private final DbConfigFactory dbConfigFactory;

    /**
     * Предполагается, что номера шардов в dbconfig - натуральные числа.
     * Номера реплик в dbconfig - тоже натуральные числа.
     * <p>
     * От номеров шардов отнимается 1, чтобы они начинались с 0.
     * <p>
     * Так как главная реплика указана в корне dbconfig-элемента, то она и записывается как нулевая реплика. А остальные
     * идентификаторы реплик используются без модификаций.
     */
    private final String[][] shardToReplicaToDbConfigPath;
    private final AtomicIntegerArray shardToRoundRobinReplica;
    private final AtomicInteger roundRobinShard;

    /**
     * Кеш доступности clickhouse. Ключ - jdbc url, значение - true если хост доступен, false если лежит.
     */
    private final Cache<String, Boolean> jdbcUrlIsUpCache;

    private DbConfigClickHouseManager(DbConfigFactory dbConfigFactory,
                                      DatabaseWrapperProvider databaseWrapperProvider,
                                      String[][] shardToReplicaToDbConfigPath,
                                      String replicaPathForAllSingleShard,
                                      Cache<String, Boolean> jdbcUrlIsUpCache) {
        this.dbConfigFactory = dbConfigFactory;
        this.databaseWrapperProvider = databaseWrapperProvider;
        this.shardToReplicaToDbConfigPath = shardToReplicaToDbConfigPath;
        this.replicaPathForAllSingleShard = replicaPathForAllSingleShard;
        this.jdbcUrlIsUpCache = jdbcUrlIsUpCache;
        this.shardToRoundRobinReplica = new AtomicIntegerArray(shardToReplicaToDbConfigPath.length);
        this.roundRobinShard = new AtomicInteger();
    }

    /**
     * @param dbConfigRootPath Путь в dbconfig, в котором описано дерево, описывающее clickhouse-кластер.
     */
    public static DbConfigClickHouseManager create(DbConfigFactory dbConfigFactory, String dbConfigRootPath,
                                                   DatabaseWrapperProvider databaseWrapperProvider) {
        String[][] shardToReplicaToDbConfigPath = extractShardReplicaMap(dbConfigFactory, dbConfigRootPath);
        Cache<String, Boolean> jdbcUrlIsUpCache = CacheBuilder.newBuilder()
                .expireAfterWrite(10, TimeUnit.SECONDS)
                .build();
        String replicaForAllSingleShard =
                chooseReplicaForAllSingleShard(dbConfigFactory, shardToReplicaToDbConfigPath[0], jdbcUrlIsUpCache);
        return new DbConfigClickHouseManager(
                dbConfigFactory,
                databaseWrapperProvider,
                shardToReplicaToDbConfigPath,
                replicaForAllSingleShard,
                jdbcUrlIsUpCache);
    }

    private static int getMainReplica(Map<Integer, ?> shardMap) {
        return Collections.min(shardMap.keySet());
    }

    /**
     * Заранее выбираются реплики, в которые будет ходить приложение для чтения/записи в таблицы, для которых важна
     * линеаризуемость вставки/чтения.
     * <p>
     * При этом важно, чтобы даже для разных таблиц хосты совпадали. Это может сузить количество доступных реплик, если
     * не на всех clickhouse-серверах одинаковый набор таблиц.
     */
    private static String chooseReplicaForAllSingleShard(DbConfigFactory dbConfigFactory,
                                                         String[] replicaPaths,
                                                         Cache<String, Boolean> jdbcUrlIsUpCache) {
        return Stream.of(replicaPaths)
                .filter(path -> isHostUp(jdbcUrlIsUpCache, dbConfigFactory.get(path)))
                .findFirst()
                .orElseThrow(() -> new IllegalStateException("No available shared replica in main shard"));
    }

    private static String[][] extractShardReplicaMap(DbConfigFactory dbConfigFactory, String dbConfigRootPath) {
        SortedMap<Integer, String[]> shardToReplicaPathArray = new TreeMap<>();
        List<String> validationErrors = new ArrayList<>();
        if (dbConfigFactory.getChildNames(dbConfigRootPath).contains("shards")) {
            for (String shardName : dbConfigFactory.getChildNames(dbConfigRootPath + ":shards")) {
                int shardNumber;
                try {
                    shardNumber = Integer.parseInt(shardName);
                } catch (NumberFormatException e) {
                    shardNumber = -1;
                }
                if (shardNumber <= 0) {
                    logger.warn("Shard name \"{}\" is not positive integer. User action log WILL NOT use it.",
                            shardName);
                } else {
                    String currentShardPath = dbConfigRootPath + ":shards:" + shardName;
                    shardToReplicaPathArray.put(shardNumber,
                            extractReplicaMap(dbConfigFactory, currentShardPath, validationErrors));
                }
            }
        } else {
            validationErrors.add("No " + dbConfigRootPath + ":shards found");
        }
        if (shardToReplicaPathArray.isEmpty()) {
            validationErrors.add("No shards found in " + dbConfigRootPath);
        }
        int expectedShardNumber = 1;
        for (int shardNumber : shardToReplicaPathArray.keySet()) {
            if (shardNumber != expectedShardNumber) {
                validationErrors.add("No shard with number " + expectedShardNumber);
                expectedShardNumber = shardNumber;
            }
            ++expectedShardNumber;
        }
        if (validationErrors.isEmpty()) {
            return shardToReplicaPathArray.values().toArray(new String[shardToReplicaPathArray.size()][]);
        } else {
            throw new ValidationError(Strings.join(validationErrors, '\n'));
        }
    }

    private static String[] extractReplicaMap(DbConfigFactory dbConfigFactory, String currentShardPath,
                                              List<String> validationErrors) {
        SortedMap<Integer, String> replicaPathMap = new TreeMap<>();
        try {
            dbConfigFactory.get(currentShardPath);
            replicaPathMap.put(0, currentShardPath);
        } catch (DbConfigException e) {
            validationErrors.add(String.format("Can't get main replica for %s: %s",
                    currentShardPath, e.getMessage()));
        }
        boolean hasReplicas;
        try {
            hasReplicas = dbConfigFactory.getChildNames(currentShardPath).contains("replicas");
        } catch (DbConfigException e) {
            validationErrors.add(String.format("Can't get replica list for %s: %s",
                    currentShardPath, e.getMessage()));
            hasReplicas = false;
        }
        if (hasReplicas) {
            for (String replicaName : dbConfigFactory.getChildNames(currentShardPath + ":replicas")) {
                int replicaNumber;
                try {
                    replicaNumber = Integer.parseInt(replicaName);
                } catch (NumberFormatException e) {
                    replicaNumber = -1;
                }
                if (replicaNumber <= 0) {
                    logger.warn(
                            "Replica name \"{}\" is not positive integer. User action log WILL NOT use it.",
                            replicaName);
                } else {
                    String currentReplicaPath = currentShardPath + ":replicas:" + replicaName;
                    try {
                        dbConfigFactory.get(currentReplicaPath);
                        replicaPathMap.put(replicaNumber, currentReplicaPath);
                    } catch (DbConfigException e) {
                        validationErrors.add(String.format("Can't get replica %s: %s",
                                currentReplicaPath, e.getMessage()));
                    }
                }
            }
        }
        int expectedReplicaNumber = 0;
        boolean hasErrors = false;
        for (int replicaNumber : replicaPathMap.keySet()) {
            if (replicaNumber != expectedReplicaNumber) {
                validationErrors.add(
                        "No replica with number " + expectedReplicaNumber + " in shard " + currentShardPath);
                expectedReplicaNumber = replicaNumber;
                hasErrors = true;
            }
            ++expectedReplicaNumber;
        }
        if (hasErrors) {
            return new String[0];
        } else {
            return replicaPathMap.values().toArray(new String[replicaPathMap.size()]);
        }
    }

    /**
     * Проверяет, что clickhouse-хост прямо сейчас может отвечать на запросы
     */
    static boolean isHostUp(Cache<String, Boolean> cache, DbConfig dbConfig) {
        String url = dbConfig.getJdbcUrl();
        Boolean result = cache.getIfPresent(url);
        if (result == null) {
            result = isHostUp(dbConfig);
            cache.put(url, result);
        }
        return result;
    }

    static boolean isHostUp(DbConfig dbConfig) {
        String url = dbConfig.getJdbcUrl();
        ClickHouseDataSource dataSource = new ClickHouseDataSource(url);
        try {
            try (Connection connection = dataSource.getConnection()) {
                return connection.isValid(Math.toIntExact(dbConfig.getConnectTimeout(TimeUnit.SECONDS)));
            } catch (RuntimeException e) {
                // Ошибка может быть завёрнута в RuntimeException
                Throwable cause = e.getCause();
                while (cause != null) {
                    if (cause instanceof ClickHouseException) {
                        return isHostUpHandleClickHouseException((ClickHouseException) cause);
                    } else if (cause instanceof RuntimeException) {
                        cause = cause.getCause();
                    } else {
                        break;
                    }
                }
                throw e;
            }
        } catch (ClickHouseException e) {
            return isHostUpHandleClickHouseException(e);
        } catch (SQLRecoverableException ignored) {
            logger.warn("Host is down: {}", url);
            return false;
        } catch (SQLException e) {
            throw new IllegalStateException(e);
        }
    }

    private static boolean isHostUpHandleClickHouseException(ClickHouseException e) {
        // Clickhouse превращает connection timed out и connection refused в
        // одну ошибку с кодом NETWORK_ERROR
        if (e.getErrorCode() == ClickHouseErrorCode.NETWORK_ERROR.code
                || e.getErrorCode() == ClickHouseErrorCode.TIMEOUT_EXCEEDED.code) {
            return false;
        } else {
            throw new IllegalStateException(e);
        }
    }

    @Override
    public DatabaseWrapper getForReading(String tableName) {
        // Хоть сейчас логика выбора хоста для записи и для чтения совпадает, в будущем это может измениться, и
        // рефакторить будет сложно.
        return get(tableName);
    }

    @Override
    public DatabaseWrapper getForWriting(String tableName) {
        return get(tableName);
    }

    private DatabaseWrapper get(String tableName) {
        ClusterDbLabel label = getLabelForTable(tableName);
        if (NEED_LINEARIZABLE_INSERTS.contains(label)) {
            return databaseWrapperProvider.get(replicaPathForAllSingleShard);
        } else {
            int shard = roundRobinShard.getAndUpdate(
                    number -> (number + 1) % shardToReplicaToDbConfigPath.length);
            int shardReplicaCount = shardToReplicaToDbConfigPath[shard].length;
            BitSet triedReplicas = new BitSet(shardReplicaCount);
            for (int i = 0; i < 1000 && triedReplicas.cardinality() < shardReplicaCount; ++i) {
                int replica = shardToRoundRobinReplica.getAndUpdate(shard,
                        number -> (number + 1) % shardReplicaCount);
                if (!triedReplicas.get(replica)) {
                    String replicaPath = shardToReplicaToDbConfigPath[shard][replica];
                    if (isHostUp(jdbcUrlIsUpCache, dbConfigFactory.get(replicaPath))) {
                        return databaseWrapperProvider.get(replicaPath);
                    }
                    triedReplicas.set(replica);
                }
            }
            throw new IllegalStateException("All replicas for in shard " + shard + " are not accessible");
        }
    }

    @Override
    public Collection<DatabaseWrapper> getAllShardReplicas(String tableName) {
        return Stream.of(shardToReplicaToDbConfigPath)
                .flatMap(Stream::of)
                .map(databaseWrapperProvider::get)
                .collect(Collectors.toList());
    }

    @Override
    public ClusterDbLabel getLabelForTable(String tableName) {
        switch (tableName) {
            case TableNames.READ_USER_ACTION_LOG_TABLE:
            case TableNames.WRITE_USER_ACTION_LOG_TABLE:
                return ClusterDbLabel.LOG;
            case TableNames.DICT_TABLE:
                return ClusterDbLabel.DICT;
            case TableNames.USER_ACTION_LOG_STATE_TABLE:
                return ClusterDbLabel.STATE;
            default:
                throw new UnsupportedOperationException(tableName);
        }
    }

    /**
     * Набор DbConfig для всех хостов, которые есть в кластере
     */
    @Override
    public Collection<DbConfig> allUniqueDbConfigs() {
        return Stream.of(shardToReplicaToDbConfigPath)
                .flatMap(Stream::of)
                .map(dbConfigFactory::get)
                .collect(Collectors.toList());
    }

    @Override
    public Collection<DatabaseWrapper> findEachShardLeader(String tableName, Duration timeout) {
        String sql = "SELECT `is_leader` FROM `system`.`replicas` WHERE database = ? AND table = ?";
        Completer.Builder builder = new Completer.Builder(timeout);
        Collection<DatabaseWrapper> result = new ArrayList<>();
        Object resultMutex = new Object();
        String[][] allPossibleShards;
        if (NEED_LINEARIZABLE_INSERTS.contains(getLabelForTable(tableName))) {
            allPossibleShards = new String[][]{shardToReplicaToDbConfigPath[0]};
        } else {
            allPossibleShards = shardToReplicaToDbConfigPath;
        }
        for (String[] replicaPaths : allPossibleShards) {
            for (String replica : replicaPaths) {
                String database = dbConfigFactory.get(replica).getDb();
                DatabaseWrapper wrapper = databaseWrapperProvider.get(replica);
                builder.submitVoid("findLeader:" + replica,
                        () -> wrapper.getDslContext().connection(connection -> {
                            // Если какой-то из серверов лежит, то будет брошена ошибка.
                            // Чтобы поправить эту вполне штатную ситуацию надо либо обернуть костылями clickhouse-jdbc,
                            // который то бросает ClickHouseUnknownException, то RuntimeError, либо поправить
                            // clickhouse-jdbc.
                            try (PreparedStatement statement = connection.prepareStatement(sql)) {
                                statement.setString(1, database);
                                statement.setString(2, tableName);
                                ResultSet resultSet = statement.executeQuery();
                                if (!resultSet.next()) {
                                    throw new IllegalArgumentException(
                                            "Table " + tableName + " is not ReplicatedMergeTree");
                                } else if (resultSet.getInt(1) != 0) {
                                    synchronized (resultMutex) {
                                        result.add(wrapper);
                                    }
                                }
                            }
                        }));
            }
        }
        try (Completer completer = builder.build()) {
            Interrupts.failingRun(() -> completer.waitAll(timeout));
        }
        // Если waitAll не упал, значит все реплики ответили. Нет нужды проверять, что в результате есть все шарды.
        return result;
    }

    /**
     * Бросается если dbconfig кластера невалидный
     */
    public static class ValidationError extends RuntimeException {
        ValidationError(String message) {
            super(message);
        }
    }
}
