package ru.yandex.direct.useractionlog.db;

import java.util.ArrayList;
import java.util.Collection;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;

import ru.yandex.direct.db.config.DbConfig;
import ru.yandex.direct.db.config.DbConfigFactory;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapper;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapperProvider;
import ru.yandex.direct.utils.db.MySQLConnector;

@ParametersAreNonnullByDefault
public class DbConfigUtil {
    private DbConfigUtil() {
    }

    public static Collection<String> getDbShards(DbConfigFactory factory, String dbPath) {
        if (!factory.has(dbPath)) {
            throw new IllegalArgumentException("invalid db path " + dbPath);
        }
        if (!factory.getChildNames(dbPath).contains("shards")) {
            throw new IllegalArgumentException("no shards for " + dbPath);
        }
        Collection<String> result = new ArrayList<>();
        for (String name : factory.getChildNames(dbPath + ":shards")) {
            result.add(dbPath + ":shards:" + name);
        }
        return result;
    }

    public static Collection<String> getShardReplicas(DbConfigFactory factory, String shardPath) {
        if (!factory.has(shardPath)) {
            throw new IllegalArgumentException("invalid shard path " + shardPath);
        }
        Collection<String> result = new ArrayList<>();
        result.add(shardPath);
        if (factory.getChildNames(shardPath).contains("replicas")) {
            for (String name : factory.getChildNames(shardPath + ":replicas")) {
                result.add(shardPath + ":replicas:" + name);
            }
        }
        return result;
    }

    public static Collection<String> getDbReplicas(DbConfigFactory factory, String dbPath) {
        Collection<String> result = new ArrayList<>();
        for (String shard : getDbShards(factory, dbPath)) {
            result.addAll(getShardReplicas(factory, shard));
        }
        return result;
    }

    public static DatabaseWrapper getWorkingReplica(
            DatabaseWrapperProvider provider, DbConfigFactory factory, String shardPath) {
        return getWorkingReplica(provider, getShardReplicas(factory, shardPath));
    }

    public static DatabaseWrapper getWorkingReplica(
            DatabaseWrapperProvider provider, Collection<String> replicaPaths) {
        for (String replica : replicaPaths) {
            DatabaseWrapper wrapper = provider.get(replica);
            if (wrapper.isAlive()) {
                return wrapper;
            }
        }
        throw new IllegalStateException("no working replica found");
    }

    public static DbConfig dbConfigForMysql(MySQLConnector mysql) {
        DbConfig dbConfig = new DbConfig();
        dbConfig.setEngine(DbConfig.Engine.MYSQL);
        dbConfig.setHosts(ImmutableList.of(mysql.getHost()));
        dbConfig.setPass(mysql.getPassword());
        dbConfig.setPort(mysql.getPort());
        dbConfig.setUser(mysql.getUsername());
        return dbConfig;
    }
}
