package ru.yandex.direct.useractionlog.db;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.clickhouse.ClickHouseDataSource;
import ru.yandex.direct.clickhouse.TableSchema;
import ru.yandex.direct.db.config.DbConfig;
import ru.yandex.direct.useractionlog.TableNames;
import ru.yandex.direct.useractionlog.schema.ActionLogSchema;
import ru.yandex.direct.useractionlog.schema.StateSchema;
import ru.yandex.direct.useractionlog.schema.dict.DictSchema;

/**
 * Утилитарный класс для создания схемы БД пользовательских логов.
 */
@ParametersAreNonnullByDefault
public class DbSchemaCreator {
    private final ClickHouseClusterInfo clusterInfo;
    private final String distributedLabel;
    private final String mergeTreeLabel;

    /**
     * @param clusterInfo                  Все хосты из dbconfig
     * @param distributedAndMergeTreeLabel Метка для секции macro в конфигурации clickhouse.
     *                                     См. https://clickhouse.yandex/docs/en/single/index.html#creating-replicated-tables
     *                                     и https://clickhouse.yandex/docs/en/single/index.html#distributed
     */
    public DbSchemaCreator(ClickHouseClusterInfo clusterInfo, String distributedAndMergeTreeLabel) {
        this(clusterInfo, distributedAndMergeTreeLabel, distributedAndMergeTreeLabel);
    }

    public DbSchemaCreator(ClickHouseClusterInfo clusterInfo, String distributedLabel, String mergeTreeLabel) {
        this.clusterInfo = clusterInfo;
        this.distributedLabel = distributedLabel;
        this.mergeTreeLabel = mergeTreeLabel;
    }

    public DbSchemaCreator(ClickHouseClusterInfo clusterInfo) {
        this(clusterInfo, "shard");
    }

    /**
     * Создать все схемы. Если какая-либо из схем уже создана, ошибки не будет.
     */
    public void execute() throws SQLException {
        for (DbConfig dbConfig : clusterInfo.allUniqueDbConfigs()) {
            try (Connection connection = new ClickHouseDataSource(dbConfig.getJdbcUrl()).getConnection()) {
                for (TableSchema schema : getSchemaList(dbConfig.getDb())) {
                    schema.createTable(connection, true);
                }
            }
        }
    }

    /**
     * Генерирует bash-скрипт, который создаст все таблицы во всём кластере.
     * Использует {@code curl}, а не {@code clickhouse-client}, т.к. в dbconfig указываются http-порты для серверов,
     * {@code clickhouse-client} не умеет с ними работать.
     */
    public String bashScriptForCluster() {
        StringBuilder script = new StringBuilder()
                .append("#!/bin/bash\n")
                .append("set -e");
        for (DbConfig dbConfig : clusterInfo.allUniqueDbConfigs()) {
            String curl = String.format(
                    "curl -X POST 'http://%s:%d/%s' --data-binary @- -H \"X-ClickHouse-User: %s\"",
                    dbConfig.getHosts().get(0), dbConfig.getPort(), dbConfig.getDb(), dbConfig.getUser());
            if (!org.apache.commons.lang3.StringUtils.isEmpty(dbConfig.getPass())) {
                curl += " -H \"X-ClickHouse-Key: " + dbConfig.getPass() + "\"";
            }
            for (TableSchema schema : getSchemaList(dbConfig.getDb())) {
                script.append("\n")
                        .append(curl)
                        .append("<<'END_OF_QUERY'\n")
                        .append(schema.toCreateTable(true))
                        .append("\nEND_OF_QUERY\n");
            }
        }
        return script.toString();
    }

    private List<TableSchema> getSchemaList(String dbName) {
        List<TableSchema> result = new ArrayList<>();

        ActionLogSchema readActionLog = new ActionLogSchema(dbName, TableNames.READ_USER_ACTION_LOG_TABLE);
        ActionLogSchema writeActionLog = new ActionLogSchema(dbName, TableNames.WRITE_USER_ACTION_LOG_TABLE);
        result.add(distributed(readActionLog, writeActionLog));
        result.add(replicatedMergeTree(writeActionLog));

        result.add(replicatedMergeTree(new StateSchema(dbName, TableNames.USER_ACTION_LOG_STATE_TABLE)));
        result.add(replicatedMergeTree(new DictSchema(dbName, TableNames.DICT_TABLE)));

        return result;
    }

    private TableSchema replicatedMergeTree(TableSchema schema) {
        return schema.toReplicated(
                String.format("/clickhouse/tables/{%s}/%s__%s",
                        mergeTreeLabel, schema.getDbName(), schema.getTableName()),
                "{replica}");
    }

    private TableSchema distributed(TableSchema schema, TableSchema readFromSchema) {
        return schema.toDistributed(readFromSchema.getDbName(), readFromSchema.getTableName(),
                distributedLabel, "rand()");
    }
}
