package ru.yandex.direct.useractionlog.db;

import java.sql.ResultSet;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.SingleColumnRowMapper;

import ru.yandex.direct.binlogclickhouse.schema.FieldValueList;
import ru.yandex.direct.clickhouse.ClickHouseSelectable;
import ru.yandex.direct.clickhouse.ClickHouseUtils;
import ru.yandex.direct.clickhouse.SimpleField;
import ru.yandex.direct.clickhouse.SqlBuilder;
import ru.yandex.direct.clickhouse.SqlUnionBuilder;
import ru.yandex.direct.clickhouse.types.DateTimeClickHouseType;
import ru.yandex.direct.clickhouse.types.OptionalStringClickHouseType;
import ru.yandex.direct.clickhouse.types.StringClickHouseType;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapper;
import ru.yandex.direct.tracing.data.DirectTraceInfo;
import ru.yandex.direct.tracing.util.TraceUtil;
import ru.yandex.direct.useractionlog.CampaignId;
import ru.yandex.direct.useractionlog.ClientId;
import ru.yandex.direct.useractionlog.model.AutoChangeableSettings;
import ru.yandex.direct.useractionlog.model.AutoUpdatedSettingsEvent;
import ru.yandex.direct.useractionlog.model.RecommendationsManagementHistory;
import ru.yandex.direct.useractionlog.schema.ActionLogRecord;
import ru.yandex.direct.useractionlog.schema.ActionLogSchema;
import ru.yandex.direct.useractionlog.schema.ObjectPath;
import ru.yandex.direct.useractionlog.schema.RecordSource;

import static java.util.Collections.emptyMap;
import static ru.yandex.direct.useractionlog.schema.ActionLogSchema.DATETIME;
import static ru.yandex.direct.useractionlog.schema.ActionLogSchema.METHOD;
import static ru.yandex.direct.useractionlog.schema.ActionLogSchema.NEW_FIELDS_NAMES;
import static ru.yandex.direct.useractionlog.schema.ActionLogSchema.NEW_FIELDS_VALUES;
import static ru.yandex.direct.useractionlog.schema.ActionLogSchema.OLD_FIELDS_NAMES;
import static ru.yandex.direct.useractionlog.schema.ActionLogSchema.OLD_FIELDS_VALUES;
import static ru.yandex.direct.useractionlog.schema.ActionLogSchema.PATH;

/**
 * Репозиторий с методами для чтения из таблицы {@link ActionLogSchema}.
 */
@ParametersAreNonnullByDefault
public class ReadActionLogTable {
    private static final ClickHouseSelectable[] FIELD_LIST =
            SqlMappers.columnList(SqlMappers.ACTION_LOG_RECORD_MAPPERS).toArray(new ClickHouseSelectable[0]);

    private static final List<Pair<SimpleField, Function<Offset, Object>>> FIELD_AND_OFFSET_EXTRACTOR_LIST;
    private static final String OFFSET_EXPRESSION_ASC;
    private static final String OFFSET_EXPRESSION_DESC;
    private static final ImmutableList<SqlBuilder.Column> ORDER_BY_COLUMNS;
    private static final String NEW_VALUE = "new_value";
    private static final String OLD_VALUE = "old_value";
    private static final String NEW_FIELD_VALUE_INDEX = "new_field_value_index";
    private static final String OLD_FIELD_VALUE_INDEX = "old_field_value_index";
    protected static final String AUTO_APPLY_JOB_NAME = "recommendations.RecommendationsAutoApplyJob";

    private static final SimpleField<String> REC_OPT_NAME = new SimpleField<>("rec_opt_name",
            new StringClickHouseType());
    private static final SimpleField<String> OBJECT_ITEM = new SimpleField<>("object_item",
            new StringClickHouseType());
    private static final SimpleField<String> OBJECT_SUBITEM = new SimpleField<>("object_subitem",
            new StringClickHouseType());
    private static final SimpleField<LocalDateTime> LAST_UPDATE_DATETIME =
            new SimpleField<>("last_update_datetime", new DateTimeClickHouseType());
    private static final SimpleField<Optional<String>> LAST_UPDATE_NEW_VALUE =
            new SimpleField<>("last_update_new_value", new OptionalStringClickHouseType(""));
    private static final SimpleField<Optional<String>> LAST_UPDATE_OLD_VALUE =
            new SimpleField<>("last_update_old_value", new OptionalStringClickHouseType(""));


    static {
        FIELD_AND_OFFSET_EXTRACTOR_LIST = Arrays.asList(
                Pair.of(ActionLogSchema.DATE, o -> o.getDateTime().toLocalDate()),
                Pair.of(DATETIME, Offset::getDateTime),
                Pair.of(ActionLogSchema.GTID, Offset::getGtid),
                Pair.of(ActionLogSchema.QUERY_SERIAL, Offset::getQuerySerial),
                Pair.of(ActionLogSchema.ROW_SERIAL, Offset::getRowSerial));
        OFFSET_EXPRESSION_DESC = String.format("(%s) <= (%s)",
                String.join(", ", Arrays.asList(
                        ActionLogSchema.DATE.getExpr(),
                        DATETIME.getExpr(),
                        ActionLogSchema.GTID.getExpr(),
                        ActionLogSchema.QUERY_SERIAL.getExpr(),
                        ActionLogSchema.ROW_SERIAL.getExpr())),
                String.join(", ", Collections.nCopies(5, "?")));
        OFFSET_EXPRESSION_ASC = OFFSET_EXPRESSION_DESC.replace(" <= ", " >= ");
        ImmutableList.Builder<SqlBuilder.Column> orderByColumnsBuilder = ImmutableList.builder();
        FIELD_AND_OFFSET_EXTRACTOR_LIST.forEach(
                p -> orderByColumnsBuilder.add(new SqlBuilder.Column(p.getLeft().getName())));
        ORDER_BY_COLUMNS = orderByColumnsBuilder.build();
    }

    private final Function<String, DatabaseWrapper> readDatabaseWrapperFn;
    private final String tableName;

    /**
     * @param readDatabaseWrapperFn Получение DatabaseWrapper по имени таблицы
     * @param tableName             Имя таблицы
     */
    public ReadActionLogTable(Function<String, DatabaseWrapper> readDatabaseWrapperFn, String tableName) {
        this.readDatabaseWrapperFn = readDatabaseWrapperFn;
        this.tableName = tableName;
    }

    private static ActionLogRecord fromResultSet(ResultSet rs) {
        return new ActionLogRecord(
                DATETIME.from(rs),
                PATH.from(rs),
                ActionLogSchema.GTID.from(rs),
                ActionLogSchema.QUERY_SERIAL.from(rs),
                ActionLogSchema.ROW_SERIAL.from(rs),
                new DirectTraceInfo(ActionLogSchema.REQID.from(rs),
                        ActionLogSchema.SERVICE.from(rs),
                        ActionLogSchema.METHOD.from(rs),
                        ActionLogSchema.OPERATOR_UID.from(rs)),
                ActionLogSchema.DB.from(rs),
                ActionLogSchema.TYPE.from(rs),
                ActionLogSchema.OPERATION.from(rs),
                FieldValueList.zip(ActionLogSchema.OLD_FIELDS_NAMES.from(rs),
                        OLD_FIELDS_VALUES.from(rs),
                        ActionLogSchema.OLD_FIELDS_NULLITIES.from(rs)),
                FieldValueList.zip(NEW_FIELDS_NAMES.from(rs),
                        NEW_FIELDS_VALUES.from(rs),
                        ActionLogSchema.NEW_FIELDS_NULLITIES.from(rs)),
                new RecordSource(ActionLogSchema.RECORD_SOURCE_TYPE.from(rs),
                        ActionLogSchema.RECORD_SOURCE_TIMESTAMP.from(rs)),
                ActionLogSchema.DELETED.from(rs));
    }

    @SuppressWarnings("unchecked")
    private static Object[] extractBindingsFromOffset(Offset offset) {
        Object[] result = new Object[FIELD_AND_OFFSET_EXTRACTOR_LIST.size()];
        for (int i = 0; i < result.length; ++i) {
            ClickHouseSelectable field = FIELD_AND_OFFSET_EXTRACTOR_LIST.get(i).getLeft();
            Function<Offset, Object> extractor = FIELD_AND_OFFSET_EXTRACTOR_LIST.get(i).getRight();
            result[i] = field.getType().toSqlObject(extractor.apply(offset));
        }
        return result;
    }

    /**
     * @param offset Если есть - смещение по дате, gtid, querySerial, rowSerial.
     * @param order  Порядок сортировки, единый для всех полей.
     * @return Генератор запросов для {@link #select(SqlBuilder)}, в котором уже выбраны все необходимые поля и название
     * таблицы, а также сортировка по убыванию по дате, gtid, querySerial, rowSerial.
     */
    public SqlBuilder sqlBuilderWithSort(@Nullable Offset offset, Order order) {
        SqlBuilder sqlBuilder = new SqlBuilder()
                .select(FIELD_LIST)
                .from(tableName);
        for (SqlBuilder.Column columnName : ORDER_BY_COLUMNS) {
            sqlBuilder.orderBy(
                    columnName,
                    order == Order.ASC ? SqlBuilder.Order.ASC : SqlBuilder.Order.DESC);
        }
        if (offset != null) {
            sqlBuilder.where(
                    order == Order.ASC ? OFFSET_EXPRESSION_ASC : OFFSET_EXPRESSION_DESC,
                    extractBindingsFromOffset(offset));
        }
        return sqlBuilder;
    }

    /**
     * Получить все записи, соответствующие запросу.
     *
     * @param sqlBuilder Генератор запросов, который был возвращён методом {@link #sqlBuilderWithSort(Offset, Order)}.
     */
    public List<ActionLogRecord> select(SqlBuilder sqlBuilder) {
        return selectWithTraceComment(sqlBuilder, rs -> {
            List<ActionLogRecord> result = new ArrayList<>();
            while (rs.next()) {
                result.add(fromResultSet(rs));
            }
            return result;
        });
    }

    private <R> List<R> selectWithTraceComment(SqlBuilder sqlBuilder, ResultSetExtractor<List<R>> extractor){
        return readDatabaseWrapperFn.apply(tableName).query(jdbc -> {
            SqlBuilder traceSqlBuilder = sqlBuilder.withComment(TraceUtil.getTraceSqlComment());
            return jdbc.query(traceSqlBuilder.toString(), traceSqlBuilder.getBindings(), extractor);
        });
    }

    /**
     * Этот запрос вернёт для каждого дня приблизительное количество записей, соответствующих фильтру.
     * Возвращаемое количество всегда будет больше либо равно реальному количеству по следующим причинам:
     * <ul>
     * <li>В целях ускорения не используется конструкция {@code SELECT FINAL}.</li>
     * <li>Если указан {@code offset}, то в целях ускорения из него будет взята только дата, а не весь кортеж.</li>
     * </ul>
     *
     * @param offset Если есть - смещение по дате.
     * @param order  Указывает на то, какую часть отрезает смещение. В случае {@code DESC} отрезает всё, что свежее
     *               {@code offset}, в случае {@code ASC} - всё, что старее.
     * @return Генератор запросов для {@link #getCountByDate(SqlBuilder)}, в котором уже выбраны все необходимые поля и
     * название таблицы, а также сортировка по убыванию по дате, gtid, querySerial, rowSerial.
     */
    public SqlBuilder dateAndCountSqlBuilder(@Nullable Offset offset, Order order) {
        SqlBuilder sqlBuilder = new SqlBuilder()
                .select(ActionLogSchema.DATE.getName())
                .selectExpression("count()", "count")
                .from(tableName)
                .groupBy(ActionLogSchema.DATE.getName());
        if (offset != null) {
            sqlBuilder.where(SqlBuilder.column(ActionLogSchema.DATE.getName()),
                    order == Order.ASC ? ">=" : "<=",
                    ActionLogSchema.DATE.getType().toSqlObject(offset.getDateTime().toLocalDate()));
        }
        return sqlBuilder;
    }

    public static SqlBuilder objectPathSqlBuilder(SqlBuilder builder, Collection<ObjectPath> objectPaths){
        List<Object> pathValues = new ArrayList<>();
        for (ObjectPath path : objectPaths) {
            pathValues.add(path.toPathString() + "%");
        }

        String pathQuery = PATH.getExpr() + " LIKE ?";
        return builder.where("(" + String.join(" OR ", Collections.nCopies(pathValues.size(),
                        pathQuery)) + ")", pathValues.toArray());
    }

    public List<ObjectPath> buildObjectPathsByCids(Long clientId, Collection<Long> cids){
        var clientIdObj = new ClientId(clientId);
        return cids.stream()
                .map(id -> new ObjectPath.CampaignPath(clientIdObj, new CampaignId(id)))
                .collect(Collectors.toList());
    }

    /**
     * Получение последних событий о включении-выключении настройки автоприменения рекомендаций на кампаниях
     * Пример ответа:
     *
     ┌─path───────────────────────────┬─rec_opt_name─────────────────────────────┬─last_update_new_value─┬─last_update_datetime─┐
     │ client:64938245-camp:57254816- │ price_recommendations_management_enabled │                     0 │  2022-06-24 17:12:24 │
     │ client:64938245-camp:57254816- │ price_recommendations_management_enabled │                     1 │  2022-06-14 00:53:36 │
     │ client:64938245-camp:57254816- │ recommendations_management_enabled       │                     1 │  2022-06-24 17:12:24 │
     │ client:64938245-camp:65126691- │ price_recommendations_management_enabled │                     1 │  2022-06-14 18:55:10 │
     │ client:64938245-camp:65126691- │ price_recommendations_management_enabled │                     0 │  2022-06-14 00:51:50 │
     │ client:64938245-camp:65126691- │ recommendations_management_enabled       │                     1 │  2022-06-14 18:55:10 │
     │ client:64938245-camp:65126691- │ recommendations_management_enabled       │                     0 │  2022-06-01 20:04:46 │
     └────────────────────────────────┴──────────────────────────────────────────┴───────────────────────┴──────────────────────┘
     */
    public List<RecommendationsManagementHistory> selectRecommendationManagementHistory(List<ObjectPath> campaignPaths,
                                                                                        LocalDateTime fromDate,
                                                                                        Collection<AutoChangeableSettings> input){
        var sqlBuilder = recommendationManagementHistorySqlBuilder(input, campaignPaths, fromDate);

        return selectWithTraceComment(sqlBuilder, rs -> {
            List<RecommendationsManagementHistory> result = new ArrayList<>();
            while (rs.next()) {
                result.add(new RecommendationsManagementHistory(
                        REC_OPT_NAME.from(rs),
                        PATH.from(rs),
                        LAST_UPDATE_DATETIME.from(rs),
                        LAST_UPDATE_NEW_VALUE.from(rs).orElse("")
                        )
                );
            }
            return result;
        });
    }

    /**
     * Пример запроса:
     *
     * SELECT
     *     path,
     *     rec_opt_name,
     *     new_value AS last_update_new_value,
     *     argMax(datetime, datetime) AS last_update_datetime
     * FROM
     * (
     *         SELECT
     *             path,
     *             datetime,
     *             'price_recommendations_management_enabled' AS rec_opt_name,
     *             toString(has(splitByChar(',', arrayElement(`new_fields`.`value`, arrayFirstIndex(f -> (f in ('opts')),
     *             `new_fields`.`name`))), 'price_recommendations_management_enabled')) AS new_value
     *         FROM user_action_log
     *         WHERE (path like 'client:64938245-camp:65126691-%' OR path like 'client:64938245-camp:57254816-%')
     *         AND `datetime` >= '2021-06-13'
     *         AND `type` = 'campaigns'
     *         AND arrayExists(f -> (f in ('opts')), `new_fields`.`name`)
     *
     *         UNION ALL
     *
     *         SELECT
     *             path,
     *             datetime,
     *             'recommendations_management_enabled' AS rec_opt_name,
     *             toString(has(splitByChar(',', arrayElement(`new_fields`.`value`, arrayFirstIndex(f -> (f in ('opts')),
     *             `new_fields`.`name`))), 'recommendations_management_enabled')) AS new_value
     *         FROM user_action_log
     *         WHERE (path like 'client:64938245-camp:65126691-%' OR path like 'client:64938245-camp:57254816-%')
     *         AND `datetime` >= '2021-06-13'
     *         AND `type` = 'campaigns'
     *         AND arrayExists(f -> (f in ('opts')), `new_fields`.`name`)
     * )
     * GROUP BY path, rec_opt_name, new_value
     */
    public SqlBuilder recommendationManagementHistorySqlBuilder(Collection<AutoChangeableSettings> settings,
                                                                List<ObjectPath> objectPaths,
                                                                LocalDateTime fromDate){
        return new SqlBuilder()
                .select(PATH, REC_OPT_NAME)
                .selectExpression(argMax(DATETIME.getName(), DATETIME.getName()), LAST_UPDATE_DATETIME.getName())
                .selectExpression(NEW_VALUE, LAST_UPDATE_NEW_VALUE.getName())
                .from(
                        recTypesHistorySqlBuilder(settings, objectPaths, fromDate)
                )
                .groupBy(PATH, REC_OPT_NAME, LAST_UPDATE_NEW_VALUE);
    }

    private SqlUnionBuilder recTypesHistorySqlBuilder(Collection<AutoChangeableSettings> settings,
                                                      List<ObjectPath> objectPaths,
                                                      LocalDateTime fromDate){
        var queries = StreamEx.of(settings)
                .distinct(AutoChangeableSettings::getRecommendationOptionName)
                .map(s -> recTypesHistorySqlBuilder(s, objectPaths, fromDate))
                .collect(Collectors.toList());

        return new SqlUnionBuilder(queries, true);
    }

    private SqlBuilder recTypesHistorySqlBuilder(AutoChangeableSettings settings, List<ObjectPath> objectPaths,
                                                 LocalDateTime fromDate){
        SqlBuilder builder = new SqlBuilder()
                .select(PATH, DATETIME)
                .selectExpression(settings.getRecommendationOptionName(), REC_OPT_NAME.getName(), true)
                .selectExpression(hasCampaignOption(settings.getRecommendationOptionName()), NEW_VALUE)
                .from(tableName)
                .where(SqlBuilder.column(DATETIME.getName()), ">=",
                        DATETIME.getType().toSqlObject(fromDate))
                .where(isOptionsChanged(settings.getType()));

        return objectPathSqlBuilder(builder, objectPaths);
    }

    /**
     * Проверяет наличие опции кампании (recOptionName) в общем массиве опций после изменения.
     * Все опции хранятся строкой, где в качестве разделителя используется запятая. Соответственно, сначала превращаем
     * строку в массив, а после этого уже проверяем наличие строки с именем опции в нем.
     * Делаем явный каст к String, т.к. функция has возващает Long (0 или 1).
     * Пример запроса:
     * toString(
     *  has(
     *      splitByChar(',', arrayElement(`new_fields`.`value`, arrayFirstIndex(f -> (f in ('opts')), `new_fields`.`name`))),
     *      'price_recommendations_management_enabled'
     *     )
     * )
     */
    private static String hasCampaignOption(String recOptionName){
        return "toString(has(splitByChar(',' , " + arrayElement(NEW_FIELDS_VALUES.getExpr(),
                arrayFirstIndex("opts", NEW_FIELDS_NAMES.getExpr())) + "), '" + recOptionName + "'))";
    }

    /**
     * @param objectPaths список путей объектов (например, кампаний) для поиска в логах.
     *                    С целью оптимизации предполагается что из этого списка как минимум
     *                    уже были отфильтрованы кампании где настройка автоприменения рекомендаций отключена
     * @param fromDate глубина поиска, указывается главным образом для оптимизации времени выполнения запроса
     * @param input список объектов с информацией о полях для которых возможно автоматическое изменение
     *              (например, настройка недельного бюджета автостратегии, средняя цена клика и т.п.)
     * @return список последних автоматических изменений каждого объекта переданного в objectPaths и каждого его поля
     * переданного в input
     */
    public List<AutoUpdatedSettingsEvent> selectLastAutoUpdatedSettings(List<ObjectPath> objectPaths,
                                                                        LocalDateTime fromDate,
                                                                        Collection<AutoChangeableSettings> input){
        var sqlBuilder = lastAutoUpdatedSettingsSqlBuilder(input, objectPaths, fromDate);

        return selectWithTraceComment(sqlBuilder, rs -> {
            List<AutoUpdatedSettingsEvent> result = new ArrayList<>();
            while (rs.next()) {
                result.add(new AutoUpdatedSettingsEvent()
                        .withPath(PATH.from(rs))
                        .withItem(OBJECT_ITEM.from(rs))
                        .withSubitem(OBJECT_SUBITEM.from(rs))
                        .withOldValue(LAST_UPDATE_OLD_VALUE.from(rs).orElse(null))
                        .withNewValue(LAST_UPDATE_NEW_VALUE.from(rs).orElse(null))
                        .withLastAutoUpdateTime(LAST_UPDATE_DATETIME.from(rs))
                );
            }
            return result;
        });
    }

    /**
     * Пример запроса на выходе:
     *
     * -- На верхнем уровне мы группируем записи по кампаниям (path), а также настройкам к которым были применены
     * -- изменения. В данном примере object_item будут 'day_budget' и 'strategy_data', а object_subitem соответственно
     * это null (в случае дневного бюджета), а также 'avg_bid' и 'sum' для настроек автобюджета
     *
     * SELECT
     *     path,
     *     object_item,
     *     object_subitem,
     *     argMax(datetime, datetime) AS last_update_datetime,
     *     argMax(new_value, datetime) AS last_update_new_value,
     *     argMax(old_value, datetime) AS last_update_old_value
     * FROM
     * (
     *
     *     -- На втором уровне вложенности мы непосредственно достаем значение до изменения и после изменения
     *     -- Ситуация осложняется тем что значение может лежать внутри JSON, см. метод getValueFromItemOrSubitem
     *     -- На выходе оставляем только те строки где старое и новое значения отличаются, см. WHERE
     *
     *     SELECT
     *         path,
     *         object_item,
     *         object_subitem,
     *         datetime,
     *         if(object_subitem is null, arrayElement(`new_fields`.`value`, new_field_value_index),
     *         JSONExtractRaw(arrayElement(`new_fields`.`value`, new_field_value_index) , object_subitem)) AS new_value,
     *         if(object_subitem is null, arrayElement(`old_fields`.`value`, old_field_value_index),
     *         JSONExtractRaw(arrayElement(`old_fields`.`value`, old_field_value_index), object_subitem)) AS old_value
     *     FROM
     *     (
     *
     *         --- На нижнем уровне вложенности мы делаем по SELECT'у на каждый вид изменения настроек кампании
     *         --- Например, изменение дневного бюджета или изменение настроек недельного бюджета автостратегии
     *         --- При этом в качестве WHERE мы указываем все кампании, то есть количество SELECT'ов определяется
     *         --- количеством изменяемых полей, а не количеством кампаний
     *
     *         SELECT
     *             path,
     *             datetime,
     *             'strategy_data' AS object_item,
     *             'sum' AS object_subitem,
     *             `new_fields`.`value`,
     *             `old_fields`.`value`,
     *             arrayFirstIndex(f -> (f in ('strategy_data')), `new_fields`.`name`) AS new_field_value_index,
     *             arrayFirstIndex(f -> (f in ('strategy_data')), `old_fields`.`name`) AS old_field_value_index
     *         FROM user_action_log
     *         WHERE (path like 'client:64938245-camp:65126691-%' OR path like 'client:64938245-camp:57254816-%')
     *         AND `type` = 'campaigns' AND (arrayExists(f -> (f in ('strategy_data')),
     *             if(`operation` = 'INSERT',  `new_fields`.`name`, `old_fields`.`name`)))
     *         AND method = 'recommendations.RecommendationsAutoApplyJob'
     *         AND `datetime` >= 2021-05-30
     *
     *         UNION ALL
     *
     *         SELECT
     *             path,
     *             datetime,
     *             'strategy_data' AS object_item,
     *             'avg_bid' AS object_subitem,
     *             `new_fields`.`value`,
     *             `old_fields`.`value`,
     *             arrayFirstIndex(f -> (f in ('strategy_data')), `new_fields`.`name`) AS new_field_value_index,
     *             arrayFirstIndex(f -> (f in ('strategy_data')), `old_fields`.`name`) AS old_field_value_index
     *         FROM user_action_log
     *         WHERE (path like 'client:64938245-camp:65126691-%' OR path like 'client:64938245-camp:57254816-%')
     *         AND `type` = 'campaigns' AND (arrayExists(f -> (f in ('strategy_data')),
     *             if(`operation` = 'INSERT',  `new_fields`.`name`, `old_fields`.`name`)))
     *         AND method = 'recommendations.RecommendationsAutoApplyJob'
     *         AND `datetime` >= 2021-05-30
     *
     *         UNION ALL
     *
     *         SELECT
     *             path,
     *             datetime,
     *             'day_budget' AS object_item,
     *             null AS object_subitem,
     *             `new_fields`.`value`,
     *             `old_fields`.`value`,
     *             arrayFirstIndex(f -> (f in ('day_budget')), `new_fields`.`name`) AS new_field_value_index,
     *             arrayFirstIndex(f -> (f in ('day_budget')), `old_fields`.`name`) AS old_field_value_index
     *         FROM user_action_log
     *         WHERE (path like 'client:64938245-camp:65126691-%' OR path like 'client:64938245-camp:57254816-%')
     *         AND `type` = 'campaigns' AND (arrayExists(f -> (f in ('day_budget')),
     *             if(`operation` = 'INSERT',  `new_fields`.`name`, `old_fields`.`name`)))
     *         AND method = 'recommendations.RecommendationsAutoApplyJob'
     *         AND `datetime` >= 2021-05-30
     *
     *     )
     * WHERE new_value != old_value
     * )
     * GROUP BY path, object_item, object_subitem
     */
    public SqlBuilder lastAutoUpdatedSettingsSqlBuilder(Collection<AutoChangeableSettings> settings,
                                                        List<ObjectPath> objectPaths,
                                                        LocalDateTime fromDate){
        return new SqlBuilder()
                .select(PATH, OBJECT_ITEM, OBJECT_SUBITEM)
                .selectExpression(argMax(DATETIME.getName(), DATETIME.getName()), LAST_UPDATE_DATETIME.getName())
                .selectExpression(argMax(NEW_VALUE, DATETIME.getName()), LAST_UPDATE_NEW_VALUE.getName())
                .selectExpression(argMax(OLD_VALUE, DATETIME.getName()), LAST_UPDATE_OLD_VALUE.getName())
                .from(
                        new SqlBuilder()
                                .select(PATH, OBJECT_ITEM, OBJECT_SUBITEM, DATETIME)
                                .selectExpression(getValueFromItemOrSubitem(NEW_FIELDS_VALUES.getExpr(),
                                        NEW_FIELD_VALUE_INDEX), NEW_VALUE)
                                .selectExpression(getValueFromItemOrSubitem(OLD_FIELDS_VALUES.getExpr(),
                                        OLD_FIELD_VALUE_INDEX), OLD_VALUE)
                                .from(
                                        fieldUpdatesSqlBuilder(settings, objectPaths, fromDate)
                                ).where(NEW_VALUE + " != " + OLD_VALUE)
                )
                .groupBy(PATH, OBJECT_ITEM, OBJECT_SUBITEM);
    }

    private SqlUnionBuilder fieldUpdatesSqlBuilder(Collection<AutoChangeableSettings> settings,
                                                   List<ObjectPath> objectPaths,
                                                   LocalDateTime fromDate){
        var queries = settings.stream()
                        .map(s -> fieldUpdatesSqlBuilder(s, objectPaths, fromDate))
                        .collect(Collectors.toList());

        return new SqlUnionBuilder(queries, true);
    }

    private SqlBuilder fieldUpdatesSqlBuilder(AutoChangeableSettings settings, List<ObjectPath> objectPaths,
                                          LocalDateTime fromDate){
        SqlBuilder builder = new SqlBuilder()
                .select(PATH, DATETIME, NEW_FIELDS_VALUES, OLD_FIELDS_VALUES)
                .selectExpression(settings.getItem(), OBJECT_ITEM.getName(), true)
                .selectExpression(settings.getSubitem(), OBJECT_SUBITEM.getName(), true)
                .selectExpression(arrayFirstIndex(settings.getItem(), NEW_FIELDS_NAMES.getExpr()),
                        NEW_FIELD_VALUE_INDEX)
                .selectExpression(arrayFirstIndex(settings.getItem(), OLD_FIELDS_NAMES.getExpr()),
                        OLD_FIELD_VALUE_INDEX)
                .from(tableName)
                .where(SqlBuilder.column(DATETIME.getName()), ">=",
                        DATETIME.getType().toSqlObject(fromDate))
                .where(isItemChanged(settings.getType(), settings.getItem()))
                .where(METHOD, "=", AUTO_APPLY_JOB_NAME);

        return objectPathSqlBuilder(builder, objectPaths);
    }

    /**
     * Возвращает значение поля argColumnName соответствующее строке с максимальным значением поля maxValueColumnName
     */
    private static String argMax(String argColumnName, String maxValueColumnName){
        return "argMax(" + argColumnName + ", " + maxValueColumnName + ")";
    }

    /**
     * Находит индекс первого элемента с именем objectItem в колонке columnWithArray
     */
    private static String arrayFirstIndex(String objectItem, String columnWithArray){
        return "arrayFirstIndex(f -> (f in ('" + objectItem + "')), " + columnWithArray + ")";
    }

    /**
     * Достаем из колонки содержащей массив arrayColumn элемент с индексом arrayIndex
     */
    private static String arrayElement(String arrayColumn, String arrayIndex){
        return "arrayElement(" + arrayColumn + ", " + arrayIndex + ")";
    }

    /**
     * Достает содержимое ключа key из JSON. Без приведения типа
     */
    private static String jsonExtractRaw(String jsonColumn, String key){
        return "JSONExtractRaw(" + jsonColumn + ", " + key + ")";
    }

    /**
     * Значения параметров (как до изменения - старые, так и после изменения - новые) хранятся в кликхаусе в колонках
     * `old_fields`.`value`  и `new_fields`.`value` в двух формах:
     * 1) просто как элемент массива, пример - дневной бюджет. Выглядит как-то так: ['725.00', '0', 'default']
     * 2) как вложенный JSON внутри элемента массива
     * Соответственно, чтобы извлечь значение нужно либо просто достать элемент из массива по индексу либо
     * сначала достать элемент из массива, потом распарсить JSON и лишь после этого можно достать из него значение
     * по соответствующему ключу. Для того чтобы понять как действовать в каждом конкретном случае мы смотрим
     * на значение колонки subitem, которая как раз указывает по какому ключу нужно доставать из вложенного JSON.
     * Если оно равно NULL, то значит перед нами вариант 1
     *
     * Пример получившегося запроса:
     * if(object_subitem is null,
     *          arrayElement(new_fields_value, new_field_value_index),
     *          JSONExtractRaw(arrayElement(new_fields_value, new_field_value_index) , object_subitem)
     * )
     */
    private static String getValueFromItemOrSubitem(String columnWithValue, String valueIndex){
        return ifExpression(OBJECT_SUBITEM.getName() + " is null", arrayElement(columnWithValue, valueIndex),
                jsonExtractRaw(arrayElement(columnWithValue, valueIndex), OBJECT_SUBITEM.getName()));

    }

    private static String ifExpression(String condition, String thenExpression, String elseExpression){
        return "if(" + condition + ", " + thenExpression + ", " + elseExpression + ")";
    }

    /**
     * Одно из условий по которому грепаются записи в истории изменений
     * Находим все записи подходящего типа (itemType) где одним из изменений было изменение item
     * Если это было добавление настройки (INSERT), то смотрим на поле после изменение (new_fields.name)
     * Если это было изменение (UPDATE) или удаление, то смотрим на поле до изменения (old_fields.name)
     * Например, переход с автостратегии на ручное управление приведет к тому что в поле после изменений уже не будет
     * никаких настроек автостратегий.
     * Пример получившегося запроса:
     *
     * `type` = 'campaigns' AND (arrayExists(f -> (f in ('strategy_data')),
     *             if(`operation` = 'INSERT',  `new_fields`.`name`, `old_fields`.`name`)))
     */
    private static SqlBuilder.ExpressionWithBinds isItemChanged(String itemType, String itemName){
        String expr = "type = ? AND (arrayExists(f -> (f in (?)), " +
                "if(`operation` = 'INSERT',  " + NEW_FIELDS_NAMES.getExpr() +
                ", " + OLD_FIELDS_NAMES.getExpr() + ")))";

        return new SqlBuilder.ExpressionWithBinds(expr, itemType, itemName);
    }

    /**
     * Проверяем менялись ли опции кампании. Пример запроса:
     * `type` = 'campaigns' AND arrayExists(f -> (f in ('opts')), `new_fields`.`name`)
     */
    private static SqlBuilder.ExpressionWithBinds isOptionsChanged(String itemType){
        String expr = "type = ? AND arrayExists(f -> (f in ('opts')), " + NEW_FIELDS_NAMES.getExpr() + ")";

        return new SqlBuilder.ExpressionWithBinds(expr, itemType);
    }

    /**
     * Получить количество записей для каждого дня
     *
     * @param sqlBuilder Объект, полученный в {@link #dateAndCountSqlBuilder(Offset, Order)} и к которому применили все
     *                   нужные фильтры.
     * @return Список пар дата-количество.
     */
    public List<DateAndCount> getCountByDate(SqlBuilder sqlBuilder) {
        return readDatabaseWrapperFn.apply(tableName).query(jdbc -> {
            SqlBuilder traceSqlBuilder = sqlBuilder.withComment(TraceUtil.getTraceSqlComment());
            return jdbc.query(traceSqlBuilder.toString(), traceSqlBuilder.getBindings(), rs -> {
                List<DateAndCount> result = new ArrayList<>();
                while (rs.next()) {
                    result.add(new DateAndCount(ActionLogSchema.DATE.from(rs), rs.getInt("count")));
                }
                return result;
            });
        });
    }

    /**
     * Получить для каждого server uuid время последней записи в логе
     *
     * @param minDate дата, начиная с которой рассматриваются записи
     * @param uuids   набор server uuid
     * @return соответствие uuid - время последней записи. Если в логе
     * не нашлось записи для данного uuid, то его не будет в ответе.
     */
    public Map<String, LocalDateTime> getLatestRecordTimeForUuids(LocalDate minDate, Collection<String> uuids) {
        if (uuids.isEmpty()) {
            return emptyMap();
        }
        String sql = String.format(
                "SELECT `%1$s`, max(`%2$s`) AS `%2$s`\n"
                        + "FROM (\n"
                        + "  SELECT splitByChar(':', `%3$s`)[1] AS `%1$s`, `%2$s`\n"
                        + "  FROM `%4$s`\n"
                        + "  WHERE `%5$s` >= ? AND `%1$s` IN (%6$s)\n"
                        + ")\n"
                        + "GROUP BY `%1$s`",
                "uuid",
                DATETIME.getName(),
                ActionLogSchema.GTID.getName(),
                tableName,
                ActionLogSchema.DATE.getName(),
                String.join(", ", Collections.nCopies(uuids.size(), "?")));
        List<Object> bindings = new ArrayList<>();
        bindings.add(ActionLogSchema.DATE.getType().toSqlObject(minDate));
        bindings.addAll(uuids);

        return readDatabaseWrapperFn.apply(tableName).query(jdbc -> jdbc.query(sql, bindings.toArray(), rs -> {
            Map<String, LocalDateTime> result = new HashMap<>();
            while (rs.next()) {
                result.put(rs.getString(1), DATETIME.from(rs));
            }
            return result;
        }));
    }

    /**
     * Получить распределение записей по типу между двумя моментами времени
     */
    public List<TypeCount> getCountByTypeBetween(LocalDateTime t0, LocalDateTime t1) {
        SqlBuilder sqlBuilder = new SqlBuilder()
                .select(ActionLogSchema.TYPE.getName())
                .selectExpression("count()", "count")
                .from(tableName)
                .where(String.format("%s BETWEEN ? AND ?", ActionLogSchema.DATE.getExpr()),
                        ActionLogSchema.DATE.getType().toSqlObject(t0.toLocalDate()),
                        ActionLogSchema.DATE.getType().toSqlObject(t1.toLocalDate()))
                .where(String.format("%s BETWEEN ? AND ?", DATETIME.getExpr()),
                        DATETIME.getType().toSqlObject(t0),
                        DATETIME.getType().toSqlObject(t1))
                .groupBy(ActionLogSchema.TYPE.getName());
        return readDatabaseWrapperFn.apply(tableName).query(jdbc -> {
            SqlBuilder traceSqlBuilder = sqlBuilder.withComment(TraceUtil.getTraceSqlComment());
            return jdbc.query(traceSqlBuilder.toString(), traceSqlBuilder.getBindings(), rs -> {
                List<TypeCount> result = new ArrayList<>();
                while (rs.next()) {
                    result.add(new TypeCount(ActionLogSchema.TYPE.from(rs), rs.getInt("count")));
                }
                return result;
            });
        });
    }

    public boolean isEmpty() {
        return readDatabaseWrapperFn.apply(tableName)
                .query("SELECT 1 FROM " + ClickHouseUtils.quoteName(tableName) + " LIMIT 1",
                        new SingleColumnRowMapper<String>())
                .isEmpty();
    }

    public enum Order {
        ASC,
        DESC,
    }

    /**
     * Смещение по дате, времени, gtid, querySerial и rowSerial.
     * Кортеж (gtid, querySerial, rowSerial) уникален для каждой записи в таблице логов (при условии SELECT FINAL),
     * а дата и время нужны потому, что пользователь хочет получать логи отсортированными по дате и времени.
     */
    public interface Offset {
        LocalDateTime getDateTime();

        String getGtid();

        int getQuerySerial();

        int getRowSerial();
    }

    /**
     * Пара дата-количество.
     */
    public static class DateAndCount {
        private final LocalDate date;
        private final long count;

        public DateAndCount(LocalDate date, long count) {
            this.date = date;
            this.count = count;
        }

        public LocalDate getDate() {
            return date;
        }

        public long getCount() {
            return count;
        }
    }

    /**
     * Пара тип-количество
     */
    public static class TypeCount {
        private final String type;
        private final long count;

        public TypeCount(String type, long count) {
            this.type = type;
            this.count = count;
        }

        public String getType() {
            return type;
        }

        public long getCount() {
            return count;
        }
    }
}
