package ru.yandex.direct.useractionlog.db;

import java.sql.ResultSet;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.clickhouse.SqlBuilder;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapper;
import ru.yandex.direct.tracing.util.TraceUtil;
import ru.yandex.direct.useractionlog.schema.dict.DictRecord;
import ru.yandex.direct.useractionlog.schema.dict.DictSchema;

/**
 * Репозиторий с методами для чтения {@link DictSchema}.
 */
@ParametersAreNonnullByDefault
public class ReadDictTable {
    private final Function<String, DatabaseWrapper> readDatabaseWrapperFn;
    private final String tableName;

    /**
     * @param readDatabaseWrapperFn Получение DatabaseWrapper по имени таблицы
     * @param tableName             Имя таблицы
     */
    public ReadDictTable(Function<String, DatabaseWrapper> readDatabaseWrapperFn, String tableName) {
        this.readDatabaseWrapperFn = readDatabaseWrapperFn;
        this.tableName = tableName;
    }

    private static DictRecord fromResultSet(ResultSet resultSet) {
        return new DictRecord(DictSchema.TYPE.from(resultSet),
                DictSchema.SHARD.from(resultSet),
                DictSchema.ID.from(resultSet),
                DictSchema.VALUE.from(resultSet),
                DictSchema.LAST_UPDATED.from(resultSet));
    }

    /**
     * @return Генератор запросов, в котором уже выбраны все необходимые поля и название таблицы.
     */
    public SqlBuilder sqlBuilder() {
        return new SqlBuilder()
                .select(DictSchema.TYPE.getName(),
                        DictSchema.SHARD.getName(),
                        DictSchema.ID.getName(),
                        DictSchema.VALUE.getName(),
                        DictSchema.LAST_UPDATED.getName())
                .from(tableName);
    }

    /**
     * Получить все строки из таблицы одним SQL-запросом.
     *
     * @param sqlBuilder SQL-запрос. Обязательно должны присутствовать колонки: <ul>
     *                   <li>{@link DictSchema#TYPE}</li>
     *                   <li>{@link DictSchema#SHARD}</li>
     *                   <li>{@link DictSchema#ID}</li>
     *                   <li>{@link DictSchema#VALUE}</li>
     *                   <li>{@link DictSchema#LAST_UPDATED}</li>
     *                   </ul>
     * @return Список строк
     */
    public List<DictRecord> select(SqlBuilder sqlBuilder) {
        return readDatabaseWrapperFn.apply(tableName).query(jdbc -> {
            SqlBuilder traceSqlBuilder = sqlBuilder.withComment(TraceUtil.getTraceSqlComment());
            return jdbc.query(traceSqlBuilder.toString(), traceSqlBuilder.getBindings(), rs -> {
                List<DictRecord> result = new ArrayList<>();
                while (rs.next()) {
                    result.add(fromResultSet(rs));
                }
                return result;
            });
        });
    }
}
