package ru.yandex.direct.useractionlog.db;

import java.sql.ResultSet;
import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.OptionalLong;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.clickhouse.SqlBuilder;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapper;
import ru.yandex.direct.dbutil.wrapper.SimpleDb;
import ru.yandex.direct.tracing.util.TraceUtil;
import ru.yandex.direct.useractionlog.schema.ActionLogRecord;
import ru.yandex.direct.useractionlog.schema.PpclogApiRecord;
import ru.yandex.direct.useractionlog.schema.PpclogApiSchema;

import static ru.yandex.direct.utils.DateTimeUtils.MSK;

/**
 * Репозиторий с методами для чтения {@link ru.yandex.direct.useractionlog.schema.PpclogApiSchema}
 */
@ParametersAreNonnullByDefault
public class ReadPpclogApiTable {
    // время в ppclog_api может отличаться от времени в user_action_log.
    // Для 99.9% запросов эта разница меньше минуты, но надо помнить, что
    // запись может не найтись в этом диапазоне, и тогда мы покажем
    // изменение конкретного api-приложения (e.g. DirectCommander) просто
    // как api-приложение.
    private static final Duration PAST_DELTA = Duration.ofSeconds(10);
    private static final Duration FUTURE_DELTA = Duration.ofMinutes(1);

    public static final ZoneId USER_LOG_DATE_ZONE_ID = ZoneId.of("UTC");
    private static final ZoneId PPCLOG_API_DATE_ZONE_ID = MSK;

    private static final String DATE_QUERY = String.format("%s = ?", PpclogApiSchema.LOG_DATE.getExpr());
    private static final String TIME_QUERY = String.format("%s BETWEEN ? AND ?", PpclogApiSchema.LOG_TIME.getExpr());
    private static final String REQID_QUERY_TEMPLATE = String.format("%s in (%%s)", PpclogApiSchema.REQID.getExpr());

    private final Function<String, DatabaseWrapper> dbProvider;
    private final String tableName;

    public ReadPpclogApiTable(Function<String, DatabaseWrapper> dbProvider, String tableName) {
        this.dbProvider = dbProvider;
        this.tableName = tableName;
    }

    private static PpclogApiRecord fromResultSet(ResultSet resultSet) {
        return new PpclogApiRecord(
                PpclogApiSchema.LOG_DATE.from(resultSet),
                PpclogApiSchema.LOG_TIME.from(resultSet),
                PpclogApiSchema.REQID.from(resultSet),
                PpclogApiSchema.APPLICATION_ID.from(resultSet));
    }

    /**
     * @return Генератор запросов, в котором уже выбраны все необходимые поля и название таблицы.
     */
    public SqlBuilder sqlBuilder() {
        return new SqlBuilder()
                .select(
                        PpclogApiSchema.LOG_DATE,
                        PpclogApiSchema.LOG_TIME,
                        PpclogApiSchema.REQID,
                        PpclogApiSchema.APPLICATION_ID)
                .from(tableName);
    }


    /**
     * По набору {@link ActionLogRecord} сделать WHERE условие для запроса в ppclog_api
     */
    public void applyTimeReqidFilter(SqlBuilder builder, List<ActionLogRecord> records) {
        Map<LocalDate, List<Pair<LocalDateTime, LocalDateTime>>> periods = new HashMap<>();
        Map<LocalDate, Set<Long>> reqids = new HashMap<>();
        for (ActionLogRecord record : records) {
            OptionalLong reqid = record.getDirectTraceInfo().getReqId();
            if (!reqid.isPresent()) {
                continue;
            }

            LocalDateTime timeFrom = record.getDateTime().minus(PAST_DELTA);
            LocalDateTime timeTo = record.getDateTime().plus(FUTURE_DELTA);

            // в ppclog_api время пишется в UTC (как и в пользологах), а дата
            // пишется в MSK.
            LocalDate dateFrom = timeFrom
                    .atZone(USER_LOG_DATE_ZONE_ID)
                    .withZoneSameInstant(PPCLOG_API_DATE_ZONE_ID)
                    .toLocalDate();
            LocalDate dateTo = timeTo
                    .atZone(USER_LOG_DATE_ZONE_ID)
                    .withZoneSameInstant(PPCLOG_API_DATE_ZONE_ID)
                    .toLocalDate();
            if (dateFrom.equals(dateTo)) {
                periods.computeIfAbsent(dateFrom, ignored -> new ArrayList<>())
                        .add(Pair.of(timeFrom, timeTo));
                reqids.computeIfAbsent(dateFrom, ignored -> new HashSet<>())
                        .add(reqid.getAsLong());
            } else {
                LocalDateTime startOfDateTo = dateTo
                        .atStartOfDay()
                        .atZone(PPCLOG_API_DATE_ZONE_ID)
                        .withZoneSameInstant(USER_LOG_DATE_ZONE_ID)
                        .toLocalDateTime();
                periods.computeIfAbsent(dateFrom, ignored -> new ArrayList<>())
                        .add(Pair.of(timeFrom, startOfDateTo));
                periods.computeIfAbsent(dateTo, ignored -> new ArrayList<>())
                        .add(Pair.of(startOfDateTo, timeTo));
                reqids.computeIfAbsent(dateFrom, ignored -> new HashSet<>())
                        .add(reqid.getAsLong());
                reqids.computeIfAbsent(dateTo, ignored -> new HashSet<>())
                        .add(reqid.getAsLong());
            }
        }
        if (periods.isEmpty()) {
            builder.where("0");
            return;
        }
        applyTimeReqidFilter(builder, periods, reqids);
    }

    /**
     * Добавление в {@link SqlBuilder} условий вида
     * log_date = ... AND (log_time BETWEEN t1 AND t2 OR ...) AND reqid IN (...)
     */
    public void applyTimeReqidFilter(SqlBuilder builder,
                                     Map<LocalDate, List<Pair<LocalDateTime, LocalDateTime>>> periods,
                                     Map<LocalDate, Set<Long>> reqids) {
        if (!periods.keySet().equals(reqids.keySet())) {
            throw new IllegalStateException("date sets mismatch");
        }

        List<String> dateQueries = new ArrayList<>();
        List<Object> bindings = new ArrayList<>();
        for (Map.Entry<LocalDate, List<Pair<LocalDateTime, LocalDateTime>>> datePeriods : periods.entrySet()) {
            StringBuilder whereQueryBuilder = new StringBuilder()
                    .append(DATE_QUERY);
            bindings.add(PpclogApiSchema.LOG_DATE.getType().toSqlObject(datePeriods.getKey()));

            whereQueryBuilder
                    .append(" AND (")
                    .append(String.join(" OR ", Collections.nCopies(datePeriods.getValue().size(), TIME_QUERY)))
                    .append(")");
            for (Pair<LocalDateTime, LocalDateTime> interval : periods.get(datePeriods.getKey())) {
                bindings.add(PpclogApiSchema.LOG_TIME.getType().toSqlObject(interval.getLeft()));
                bindings.add(PpclogApiSchema.LOG_TIME.getType().toSqlObject(interval.getRight()));
            }
            whereQueryBuilder
                    .append(" AND ")
                    .append(String.format(
                            REQID_QUERY_TEMPLATE,
                            StreamEx.of(reqids.get(datePeriods.getKey())).map(x -> "?").joining(", ")));
            bindings.addAll(reqids.get(datePeriods.getKey()));

            dateQueries.add(whereQueryBuilder.toString());
        }
        builder.where(String.join(" OR ", dateQueries), bindings.toArray());
    }

    /**
     * Получить все строки из таблицы одним SQL-запросом.
     *
     * @param sqlBuilder SQL-запрос. Обязательно должны присутствовать колонки: <ul>
     *                   <li>{@link PpclogApiSchema#LOG_DATE}</li>
     *                   <li>{@link PpclogApiSchema#LOG_TIME}</li>
     *                   <li>{@link PpclogApiSchema#REQID}</li>
     *                   <li>{@link PpclogApiSchema#APPLICATION_ID}</li>
     *                   </ul>
     * @return Список строк
     */
    public List<PpclogApiRecord> select(SqlBuilder sqlBuilder) {
        return dbProvider.apply(SimpleDb.CLICKHOUSE_CLOUD.toString()).query(jdbc -> {
            SqlBuilder traceSqlBuilder = sqlBuilder.withComment(TraceUtil.getTraceSqlComment());
            return jdbc.query(traceSqlBuilder.toString(), traceSqlBuilder.getBindings(), rs -> {
                List<PpclogApiRecord> result = new ArrayList<>();
                while (rs.next()) {
                    result.add(fromResultSet(rs));
                }
                return result;
            });
        });
    }
}
