package ru.yandex.direct.useractionlog.db;

import java.util.AbstractMap;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.tools.StringUtils;

import ru.yandex.direct.clickhouse.SqlBuilder;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapper;
import ru.yandex.direct.mysql.MySQLBinlogState;
import ru.yandex.direct.useractionlog.schema.StateSchema;
import ru.yandex.direct.useractionlog.schema.UserActionLogStateType;
import ru.yandex.direct.utils.Checked;
import ru.yandex.direct.utils.JsonUtils;

@ParametersAreNonnullByDefault
public class ReadStateTable implements StateReader {
    private final Function<String, DatabaseWrapper> readDatabaseWrapperFn;
    private final String tableName;
    private final SqlBuilder.ExpressionWithAlias argMaxStateDatetime = new SqlBuilder.ExpressionWithAlias(
            String.format("argMax(%s, %s)", StateSchema.STATE.getExpr(), StateSchema.DATETIME.getExpr()),
            StateSchema.STATE.getName());

    /**
     * @param readDatabaseWrapperFn Получение DatabaseWrapper по имени таблицы
     * @param tableName             Имя таблицы, из которой будут читаться данные
     */
    public ReadStateTable(Function<String, DatabaseWrapper> readDatabaseWrapperFn, String tableName) {
        this.readDatabaseWrapperFn = readDatabaseWrapperFn;
        this.tableName = tableName;
    }

    @Override
    public UserActionLogStates getActualStates(String source) {
        SqlBuilder sqlBuilder = new SqlBuilder()
                .select(StateSchema.TYPE)
                .selectExpression(argMaxStateDatetime)
                .from(tableName)
                .selectFinal()
                .where(StateSchema.SOURCE, "=", source)
                .groupBy(StateSchema.TYPE);
        UserActionLogStates.Builder result = UserActionLogStates.builder();
        DatabaseWrapper wrapper = readDatabaseWrapperFn.apply(tableName);
        wrapper.rawQuery(wrapper.render(sqlBuilder.toString()), sqlBuilder.getBindings(),
                Checked.consumer(resultSet -> {
                    while (resultSet.next()) {
                        UserActionLogStateType stateType = StateSchema.TYPE.from(resultSet);
                        String serializedState = StateSchema.STATE.from(resultSet);
                        MySQLBinlogState state;
                        state = deserializeMySQLBinlogState(serializedState);
                        switch (stateType) {
                            case DICT:
                                result.withDict(state);
                                break;
                            case LOG:
                                result.withLog(state);
                                break;
                            default:
                                throw new IllegalStateException(state.toString());
                        }
                    }
                }));
        return result.build();
    }

    @Nullable
    private static MySQLBinlogState deserializeMySQLBinlogState(String serializedState) {
        if (StringUtils.isEmpty(serializedState)) {
            return null;
        } else {
            return JsonUtils.fromJson(serializedState, MySQLBinlogState.class);
        }
    }

    /**
     * Актуальные состояния логов для всех источников.
     */
    @Override
    public Map<String, MySQLBinlogState> getAllActualLogStates() {
        SqlBuilder sqlBuilder = new SqlBuilder()
                .select(StateSchema.SOURCE)
                .selectExpression(argMaxStateDatetime)
                .from(tableName)
                .selectFinal()
                .where(StateSchema.TYPE, "=", StateSchema.TYPE.getType().toSqlObject(UserActionLogStateType.LOG))
                .groupBy(StateSchema.SOURCE);
        return readDatabaseWrapperFn.apply(tableName).query(sqlBuilder.toString(), sqlBuilder.getBindings(),
                (resultSet, number) -> new AbstractMap.SimpleEntry<>(
                        StateSchema.SOURCE.from(resultSet),
                        deserializeMySQLBinlogState(StateSchema.STATE.from(resultSet))))
                .stream()
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
    }
}
