package ru.yandex.direct.useractionlog.db;

import java.sql.PreparedStatement;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;

import ru.yandex.direct.clickhouse.ClickHouseInsertable;
import ru.yandex.direct.clickhouse.ClickHouseSelectable;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapper;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.useractionlog.schema.ActionLogRecord;
import ru.yandex.direct.useractionlog.schema.ActionLogSchema;
import ru.yandex.direct.useractionlog.schema.dict.DictRecord;
import ru.yandex.direct.useractionlog.schema.dict.DictSchema;

/**
 * Вспомогательные функции для отображения объектов в sql-запросы. Если бы jooq поддерживал clickhouse, то
 * потребности в этих функциях не было бы.
 */
@ParametersAreNonnullByDefault
class SqlMappers {
    static final List<FieldMapper<ActionLogRecord>> ACTION_LOG_RECORD_MAPPERS = Arrays.asList(
            new FieldMapper<>(ActionLogSchema.DATE, r -> r.getDateTime().toLocalDate()),
            new FieldMapper<>(ActionLogSchema.DATETIME, ActionLogRecord::getDateTime),
            new FieldMapper<>(ActionLogSchema.PATH, ActionLogRecord::getPath),
            new FieldMapper<>(ActionLogSchema.GTID, ActionLogRecord::getGtid),
            new FieldMapper<>(ActionLogSchema.QUERY_SERIAL, ActionLogRecord::getQuerySerial),
            new FieldMapper<>(ActionLogSchema.ROW_SERIAL, ActionLogRecord::getRowSerial),
            new FieldMapper<>(ActionLogSchema.REQID, r -> r.getDirectTraceInfo().getReqId()),
            new FieldMapper<>(ActionLogSchema.METHOD, r -> r.getDirectTraceInfo().getMethod()),
            new FieldMapper<>(ActionLogSchema.SERVICE, r -> r.getDirectTraceInfo().getService()),
            new FieldMapper<>(ActionLogSchema.OPERATOR_UID, r -> r.getDirectTraceInfo().getOperatorUid()),
            new FieldMapper<>(ActionLogSchema.DB, ActionLogRecord::getDb),
            new FieldMapper<>(ActionLogSchema.TYPE, ActionLogRecord::getType),
            new FieldMapper<>(ActionLogSchema.OPERATION, ActionLogRecord::getOperation),
            new FieldMapper<>(ActionLogSchema.OLD_FIELDS_NAMES, r -> r.getOldFields().getNames()),
            new FieldMapper<>(ActionLogSchema.OLD_FIELDS_VALUES, r -> r.getOldFields().getValuesAsNonNullStrings()),
            new FieldMapper<>(ActionLogSchema.OLD_FIELDS_NULLITIES, r -> r.getOldFields().getNullities()),
            new FieldMapper<>(ActionLogSchema.NEW_FIELDS_NAMES, r -> r.getNewFields().getNames()),
            new FieldMapper<>(ActionLogSchema.NEW_FIELDS_VALUES, r -> r.getNewFields().getValuesAsNonNullStrings()),
            new FieldMapper<>(ActionLogSchema.NEW_FIELDS_NULLITIES, r -> r.getNewFields().getNullities()),
            new FieldMapper<>(ActionLogSchema.RECORD_SOURCE_TYPE, r -> r.getRecordSource().getType()),
            new FieldMapper<>(ActionLogSchema.RECORD_SOURCE_TIMESTAMP, r -> r.getRecordSource().getTimestamp()),
            new FieldMapper<>(ActionLogSchema.DELETED, ActionLogRecord::isDeleted));

    static final List<FieldMapper<DictRecord>> DICT_MAPPERS = Arrays.asList(
            new FieldMapper<>(DictSchema.TYPE, DictRecord::getCategory),
            new FieldMapper<>(DictSchema.SHARD, DictRecord::getShard),
            new FieldMapper<>(DictSchema.ID, DictRecord::getId),
            new FieldMapper<>(DictSchema.VALUE, DictRecord::getValue),
            new FieldMapper<>(DictSchema.DATE, r -> r.getLastUpdated().toLocalDate()),
            new FieldMapper<>(DictSchema.LAST_UPDATED, DictRecord::getLastUpdated));

    private SqlMappers() {
    }

    /**
     * Шаблон для запроса <code>INSERT INTO ... (...) VALUES (?, ?, ...)</code>.
     * Возвращает строку-шаблон, которую нужно будет пропускать через {@link String#format(String, Object...)}
     * с одним аргументом: Название таблицы
     */
    static <R> String insertQueryTemplate(List<FieldMapper<R>> fieldMappers) {
        StringBuilder builder = new StringBuilder()
                .append("INSERT ")
                .append(" INTO ")
                .append("%s");  // table name
        StringBuilder paramsBuilder = new StringBuilder();
        String delimiter = "(";
        for (FieldMapper<?> mapper : fieldMappers) {
            builder.append(delimiter).append(mapper.field.getExpr());
            paramsBuilder.append(delimiter).append("?");
            delimiter = ", ";
        }
        return builder.append(") VALUES ").append(paramsBuilder.toString()).append(")").toString();
    }

    /**
     * Возвращает список объектов, которые потом можно будет подставить
     * в {@link PreparedStatement#setObject(int, Object)}.
     */
    @SuppressWarnings("unchecked")
    static <R> ImmutableList<Function<R, Object>> fieldToSqlMapperList(List<FieldMapper<R>> fieldMappers) {
        ImmutableList.Builder<Function<R, Object>> builder = ImmutableList.builder();
        for (FieldMapper<R> fieldMapper : fieldMappers) {
            builder.add(record -> fieldMapper.field.getType().toSqlObject(fieldMapper.mapper.apply(record)));
        }
        return builder.build();
    }

    static <R> void insertRecords(DatabaseWrapper databaseWrapper,
                                  String tableName,
                                  String insertQueryTemplate,
                                  List<Function<R, Object>> fieldToSqlMapperList,
                                  Collection<R> records) {
        databaseWrapper.getDslContext().connection(conn -> {
            String sql = String.format(insertQueryTemplate, tableName);
            try (PreparedStatement statement = conn.prepareStatement(sql)) {
                for (R record : records) {
                    int counter = 0;
                    for (Function<R, Object> getter : fieldToSqlMapperList) {
                        statement.setObject(++counter, getter.apply(record));
                    }
                    statement.addBatch();
                }
                try (TraceProfile ignored = Trace.current()
                        .profile("db:read", databaseWrapper.getDbname(), records.size())) {
                    statement.executeBatch();
                }
            }
        });
    }

    static <R> ImmutableList<ClickHouseSelectable> columnList(List<FieldMapper<R>> fieldMappers) {
        ImmutableList.Builder<ClickHouseSelectable> builder = ImmutableList.builder();
        for (FieldMapper<R> fieldMapper : fieldMappers) {
            builder.add(fieldMapper.field);
        }
        return builder.build();
    }

    static class FieldMapper<R> {
        final ClickHouseInsertable field;
        final Function<R, Object> mapper;

        FieldMapper(ClickHouseInsertable field, Function<R, Object> mapper) {
            this.field = field;
            this.mapper = mapper;
        }
    }
}
