package ru.yandex.direct.useractionlog.db;

import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.clickhouse.ClickHouseUtils;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapper;
import ru.yandex.direct.mysql.MySQLBinlogState;
import ru.yandex.direct.useractionlog.schema.StateSchema;
import ru.yandex.direct.useractionlog.schema.UserActionLogStateType;
import ru.yandex.direct.utils.JsonUtils;

@ParametersAreNonnullByDefault
public class WriteStateTable implements StateWriter {
    private static final Logger logger = LoggerFactory.getLogger(WriteStateTable.class);

    private static final String INSERT_QUERY_TEMPLATE = String.format(
            "INSERT INTO %%s (%s, %s, %s, %s, %s) VALUES (?, ?, ?, ?, ?)",
            StateSchema.DATE.getExpr(),
            StateSchema.DATETIME.getExpr(),
            StateSchema.SOURCE.getExpr(),
            StateSchema.STATE.getExpr(),
            StateSchema.TYPE.getExpr());
    private final Function<String, DatabaseWrapper> writeDatabaseWrapperFn;
    private final String tableName;

    /**
     * @param writeDatabaseWrapperFn Получение DatabaseWrapper по имени таблицы
     * @param tableName              Имя таблицы
     */
    public WriteStateTable(Function<String, DatabaseWrapper> writeDatabaseWrapperFn, String tableName) {
        this.writeDatabaseWrapperFn = writeDatabaseWrapperFn;
        this.tableName = tableName;
    }

    private void saveState(String source, @Nullable MySQLBinlogState state, UserActionLogStateType... types) {
        DatabaseWrapper databaseWrapper = writeDatabaseWrapperFn.apply(tableName);
        String sql = String.format(INSERT_QUERY_TEMPLATE, ClickHouseUtils.quoteName(tableName));
        LocalDateTime dateTime = ZonedDateTime.now(ZoneOffset.UTC).toLocalDateTime();
        String serializedState = state == null ? "" : JsonUtils.toJson(state);
        List<Object[]> batchArgs = new ArrayList<>();
        for (UserActionLogStateType type : types) {
            Object[] objects = new Object[]{
                    StateSchema.DATE.getType().toSqlObject(dateTime.toLocalDate()),
                    StateSchema.DATETIME.getType().toSqlObject(dateTime),
                    StateSchema.SOURCE.getType().toSqlObject(source),
                    StateSchema.STATE.getType().toSqlObject(serializedState),
                    StateSchema.TYPE.getType().toSqlObject(type)
            };
            batchArgs.add(objects);
            logger.info("write state ({}, {}, {}, {})",
                    dateTime, source, type,
                    Optional.ofNullable(state).map(MySQLBinlogState::getGtidSet).orElse("-")
            );
        }
        databaseWrapper.query(jdbc -> jdbc.batchUpdate(sql, batchArgs));
        logger.info("state written");
    }

    @Override
    public void saveLogState(String source, MySQLBinlogState state) {
        saveState(source, state, UserActionLogStateType.LOG);
    }

    @Override
    public void saveDictState(String source, MySQLBinlogState state) {
        saveState(source, state, UserActionLogStateType.DICT);
    }

    @Override
    public void saveBothStates(String source, MySQLBinlogState state) {
        saveState(source, state, UserActionLogStateType.LOG, UserActionLogStateType.DICT);
    }
}
