package ru.yandex.direct.useractionlog.dict;

import java.sql.SQLException;
import java.util.Collection;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.ThreadSafe;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Объект, предоставляющий доступ к словарным данным всех видов. Ходит в другой {@link DictRepository}
 * за данными, помимо этого хранит полученные данные в LRU-кеше. Все новые словарные данные добавляются и в кеш,
 * и в другой {@link DictRepository}.
 */
@ParametersAreNonnullByDefault
@ThreadSafe
public class CacheDictRepository implements DictRepository {
    private static final Logger logger = LoggerFactory.getLogger(CacheDictRepository.class);

    private final ReadWriteLock lock = new ReentrantReadWriteLock();
    private final Map<DictDataCategory, CachePair> cachePairByCategory;
    private final CachePair fallbackCachePair;
    private final DictRepository forwardDict;

    /**
     * @param forwardDict Другой репозиторий, из которого будут браться данные, если их нет в кеше.
     */
    public CacheDictRepository(DictRepository forwardDict) {
        this.forwardDict = forwardDict;

        cachePairByCategory = new EnumMap<>(DictDataCategory.class);

        // Все числа взяты с потолка
        cachePairByCategory.put(DictDataCategory.CAMPAIGN_PATH, new DedicatedCachePair(
                CacheBuilder.newBuilder().maximumSize(100_000).build(),
                CacheBuilder.newBuilder().maximumSize(1_000).expireAfterWrite(10, TimeUnit.MINUTES).build()));
        cachePairByCategory.put(DictDataCategory.CAMPAIGN_NAME, new DedicatedCachePair(
                CacheBuilder.newBuilder().maximumSize(1_000).build(),
                CacheBuilder.newBuilder().maximumSize(1_000).expireAfterWrite(10, TimeUnit.MINUTES).build()));
        cachePairByCategory.put(DictDataCategory.ADGROUP_PATH, new DedicatedCachePair(
                CacheBuilder.newBuilder().maximumSize(100_000).build(),
                CacheBuilder.newBuilder().maximumSize(1_000).expireAfterWrite(10, TimeUnit.MINUTES).build()));
        cachePairByCategory.put(DictDataCategory.ADGROUP_NAME, new DedicatedCachePair(
                CacheBuilder.newBuilder().maximumSize(5_000).build(),
                CacheBuilder.newBuilder().maximumSize(1_000).expireAfterWrite(10, TimeUnit.MINUTES).build()));
        cachePairByCategory.put(DictDataCategory.AD_PATH, new DedicatedCachePair(
                CacheBuilder.newBuilder().maximumSize(100_000).build(),
                CacheBuilder.newBuilder().maximumSize(1_000).expireAfterWrite(10, TimeUnit.MINUTES).build()));
        cachePairByCategory.put(DictDataCategory.AD_TITLE, new DedicatedCachePair(
                CacheBuilder.newBuilder().maximumSize(5_000).build(),
                CacheBuilder.newBuilder().maximumSize(1_000).expireAfterWrite(10, TimeUnit.MINUTES).build()));

        fallbackCachePair = new FallbackCachePair(
                CacheBuilder.newBuilder().maximumSize(20_000).build(),
                CacheBuilder.newBuilder().maximumSize(1_000).expireAfterWrite(10, TimeUnit.MINUTES).build());
    }

    /**
     * {@inheritDoc}
     * Может взять все данные из LRU-кеша, может часть или все данные передать другому репозиторию.
     *
     * @throws IllegalStateException Оборачивает {@link SQLException} при общении с ClickHouse
     */
    @Nonnull
    @Override
    public Map<DictRequest, Object> getData(Collection<DictRequest> dictRequests) {
        Set<DictRequest> forwardRequestSet = new HashSet<>();
        Map<DictRequest, Object> result = new HashMap<>();

        int valuesFromCache = 0;
        int missesFromCache = 0;

        // Реализация с ReadWriteLock не самая эффективная, тем более что в guava есть возможность атомарно читать и
        // писать в кеш пачками
        lock.readLock().lock();
        try {
            for (DictRequest request : dictRequests) {
                DictDataCategory category = request.getCategory();
                CachePair cachePair = cachePairByCategory.getOrDefault(category, fallbackCachePair);
                long id = request.getId();
                Object cachedValue = cachePair.getValueIfPresent(category, id);
                if (cachedValue != null) {
                    ++valuesFromCache;
                    result.put(request, cachedValue);
                } else if (cachePair.getMissIfPresent(category, id) == null) {
                    forwardRequestSet.add(request);
                } else {
                    ++missesFromCache;
                }
            }
        } finally {
            lock.readLock().unlock();
        }

        int newCachedValues = 0;
        int newCachedMisses = 0;
        if (!forwardRequestSet.isEmpty()) {
            Map<DictRequest, Object> forwardResponse = forwardDict.getData(forwardRequestSet);
            result.putAll(forwardResponse);

            lock.writeLock().lock();
            try {
                newCachedValues += forwardResponse.size();
                for (Map.Entry<DictRequest, Object> forwardResponseEntry : forwardResponse.entrySet()) {
                    DictDataCategory category = forwardResponseEntry.getKey().getCategory();
                    long id = forwardResponseEntry.getKey().getId();
                    CachePair cachePair = cachePairByCategory.getOrDefault(category, fallbackCachePair);
                    cachePair.putValue(category, id, forwardResponseEntry.getValue());
                    forwardRequestSet.remove(forwardResponseEntry.getKey());
                }

                newCachedMisses += forwardRequestSet.size();
                for (DictRequest request : forwardRequestSet) {
                    DictDataCategory category = request.getCategory();
                    CachePair cachePair = cachePairByCategory.getOrDefault(category, fallbackCachePair);
                    cachePair.putMiss(category, request.getId());
                }
            } finally {
                lock.writeLock().unlock();
            }
        }

        logger.debug("Got {} requests. Totally returned {} results, {} of them from cache."
                        + " Predicted {} misses in forward dictionary. Added {} new results into cache. Remembered {} misses.",
                dictRequests.size(), result.size(), valuesFromCache,
                missesFromCache, newCachedValues, newCachedMisses);
        return result;
    }

    /**
     * {@inheritDoc}
     * Добавляет словарные данные в другой репозиторий и в LRU-кеш.
     *
     * @throws IllegalStateException Оборачивает {@link SQLException} при общении с ClickHouse
     */
    @Override
    public void addData(Map<DictRequest, Object> data) {
        int invalidatedMisses = 0;
        data = new HashMap<>(data);  // Владение data не передаётся. Вызывающая сторона может использовать объект.
        for (Map.Entry<DictRequest, Object> entry : data.entrySet()) {
            DictRequest dictRequest = entry.getKey();
            Object value = entry.getValue();
            DictDataCategory category = dictRequest.getCategory();
            long id = dictRequest.getId();
            CachePair cachePair = cachePairByCategory.getOrDefault(category, fallbackCachePair);
            cachePair.putValue(category, id, value);
            if (cachePair.getMissIfPresent(dictRequest.getCategory(), dictRequest.getId()) != null) {
                cachePair.invalidateMiss(category, id);
                ++invalidatedMisses;
            }
        }
        logger.debug("Added {} fresh records in cache, invalidated {} misses", data.size(), invalidatedMisses);
        if (!data.isEmpty()) {
            forwardDict.addData(data);
        }
    }

    private abstract static class CachePair<K> {
        /**
         * Кеш для словарных объектов
         */
        private final Cache<K, Object> values;

        /**
         * Кеш для промахов. Значение игнорируется, используется как Set. Если некоторый идентификатор отсутствовал в
         * БД, то не следует много раз подряд запрашивать его снова и снова, в надежде, что он появился из ниоткуда.
         */
        private final Cache<K, Object> misses;

        CachePair(Cache<K, Object> values, Cache<K, Object> misses) {
            this.values = values;
            this.misses = misses;
        }

        protected abstract K idToKey(DictDataCategory category, long id);

        Object getValueIfPresent(DictDataCategory category, long id) {
            return values.getIfPresent(idToKey(category, id));
        }

        Object getMissIfPresent(DictDataCategory category, long id) {
            return misses.getIfPresent(idToKey(category, id));
        }

        void putValue(DictDataCategory category, long id, Object value) {
            values.put(idToKey(category, id), value);
        }

        void putMiss(DictDataCategory category, long id) {
            misses.put(idToKey(category, id), true);
        }

        void invalidateMiss(DictDataCategory category, long id) {
            misses.invalidate(idToKey(category, id));
        }
    }

    private static class DedicatedCachePair extends CachePair<Long> {
        DedicatedCachePair(Cache<Long, Object> values, Cache<Long, Object> misses) {
            super(values, misses);
        }

        @Override
        protected Long idToKey(DictDataCategory category, long id) {
            return id;
        }
    }

    private static class FallbackCachePair extends CachePair<Pair<DictDataCategory, Long>> {
        FallbackCachePair(Cache<Pair<DictDataCategory, Long>, Object> values,
                          Cache<Pair<DictDataCategory, Long>, Object> misses) {
            super(values, misses);
        }

        @Override
        protected Pair<DictDataCategory, Long> idToKey(DictDataCategory category, long id) {
            return Pair.of(category, id);
        }
    }
}
