package ru.yandex.direct.useractionlog.dict;

import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.ThreadSafe;

import com.google.common.base.Preconditions;
import com.google.common.collect.Iterators;
import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.clickhouse.SqlBuilder;
import ru.yandex.direct.useractionlog.db.ReadWriteDictTable;
import ru.yandex.direct.useractionlog.schema.dict.DictRecord;
import ru.yandex.direct.useractionlog.schema.dict.DictSchema;
import ru.yandex.direct.utils.MonotonicTime;
import ru.yandex.direct.utils.NanoTimeClock;

/**
 * Объект, позволяющий хранить словарные данные в ClickHouse-таблице {@link DictSchema} и получать их оттуда.
 * <p>
 * В этой таблице не хранится вся история изменений словарного значения, а только наиболее свежее значение.
 * <p>
 * Является конечной точкой в поиске словарных данных, т.е. если что-то не было найдено, то ни в какой другой класс
 * запрос не будет отправлен.
 */
@ParametersAreNonnullByDefault
@ThreadSafe
public class ClickHouseDictRepository implements DictRepository {
    public static final int FETCH_CHUNK_SIZE = 5_000;
    private static final Logger logger = LoggerFactory.getLogger(ClickHouseDictRepository.class);
    @Nullable
    private final String source;
    private final ReadWriteDictTable dictTable;

    /**
     * @param source    Название источника данных. Например, "ppc:1". Если указано, то все операции чтения будут возвращать
     *                  только данные, связанные с этим источником. Если null, то для каждого запроса будет возвращаться
     *                  самое свежее значение среди всех источников, но запись без явного источника запрещена.
     * @param dictTable Объект для записи и чтения таблицы словаря
     */
    public ClickHouseDictRepository(@Nullable String source, ReadWriteDictTable dictTable) {
        this.source = source;
        this.dictTable = dictTable;
    }

    @Nonnull
    @Override
    public Map<DictRequest, Object> getData(Collection<DictRequest> dictRequests) {
        List<DictRequest> sortedRequests = new ArrayList<>(dictRequests);
        sortedRequests.sort(Comparator
                .comparing(DictRequest::getCategory)
                .thenComparingLong(DictRequest::getId));
        Map<DictRequest, Object> result = new HashMap<>();
        Iterators.partition(sortedRequests.iterator(), FETCH_CHUNK_SIZE)
                .forEachRemaining(chunk -> getDataInternal(result, chunk));
        return result;
    }

    private void getDataInternal(Map<DictRequest, Object> result, List<DictRequest> dictRequests) {
        MonotonicTime procedureStart = NanoTimeClock.now();
        SqlBuilder sqlBuilder = makeSqlBuilder(dictRequests);

        MonotonicTime queryStart = NanoTimeClock.now();
        List<DictRecord> records = dictTable.read.select(sqlBuilder);
        MonotonicTime queryEnd = NanoTimeClock.now();

        Map<DictRequest, DictRecord> recordMap = Maps.newHashMapWithExpectedSize(dictRequests.size());
        for (DictRecord record : records) {
            recordMap.merge(new DictRequest(record.getCategory(), record.getId()),
                    record,
                    (l, r) -> l.getLastUpdated().isBefore(r.getLastUpdated()) ? r : l);
        }
        recordMap.forEach((dictRequest, dictRecord) -> result.put(dictRequest,
                ClickHouseDictUtils.deserializeClickHouse(
                        dictRequest.getCategory(),
                        dictRecord.getId(),
                        dictRecord.getValue())));

        MonotonicTime procedureEnd = NanoTimeClock.now();
        if (logger.isInfoEnabled()) {
            logger.info("Got {} requests. Returned {} responses."
                            + " Procedure time: {} seconds. Query time: {} seconds.",
                    dictRequests.size(), recordMap.size(),
                    procedureEnd.minus(procedureStart).toMillis() / 1e3,
                    queryEnd.minus(queryStart).toMillis() / 1e3);
        }
    }

    private SqlBuilder makeSqlBuilder(List<DictRequest> dictRequests) {
        StringBuilder whereQuery = new StringBuilder("(");
        List<Object> whereQueryBindings = new ArrayList<>();
        DictDataCategory previousCategory = null;
        String categoryDelimiter = "";
        String idDelimiter = "";
        for (DictRequest dictRequest : dictRequests) {
            if (!Objects.equals(previousCategory, dictRequest.getCategory())) {
                previousCategory = dictRequest.getCategory();
                whereQuery.append(categoryDelimiter)
                        .append(DictSchema.TYPE.getExpr())
                        .append(" = ? AND ")
                        .append(DictSchema.ID.getExpr())
                        .append(" IN (");
                whereQueryBindings.add(DictSchema.TYPE.getType().toSqlObject(previousCategory));
                idDelimiter = "";
                categoryDelimiter = ") OR ";
            }
            whereQuery.append(idDelimiter).append("?");
            whereQueryBindings.add(DictSchema.ID.getType().toSqlObject(dictRequest.getId()));
            idDelimiter = ", ";
        }
        whereQuery.append("))");

        // ClickHouse не гарантирует уникальность по первичному ключу. Забираются все записи, соответствующие желаемым
        // первичным ключам, а затем среди дубликатов оставляются записи с максимальным last_updated.
        SqlBuilder sqlBuilder = dictTable.read.sqlBuilder();
        if (source != null) {
            sqlBuilder.where(DictSchema.SHARD, "=", source);
        }
        sqlBuilder.where(whereQuery.toString(), whereQueryBindings);
        return sqlBuilder;
    }

    @Override
    public void addData(Map<DictRequest, Object> data) {
        Preconditions.checkState(source != null, "Dict repository without specified shard can't add data");
        List<Map.Entry<DictRequest, Object>> sortedData = new ArrayList<>(data.entrySet());
        sortedData.sort(Comparator
                .comparing((Map.Entry<DictRequest, Object> r) -> r.getKey().getCategory())
                .thenComparingLong((Map.Entry<DictRequest, Object> r) -> r.getKey().getId()));

        LocalDateTime now = getUtcNow();
        List<DictRecord> records = new ArrayList<>(sortedData.size());
        for (Map.Entry<DictRequest, Object> freshValue : sortedData) {
            records.add(new DictRecord(freshValue.getKey().getCategory(),
                    Objects.requireNonNull(source),
                    freshValue.getKey().getId(),
                    ClickHouseDictUtils.serializeClickHouse(freshValue.getKey().getCategory(), freshValue.getValue()),
                    now));
        }

        MonotonicTime start = NanoTimeClock.now();
        if (!records.isEmpty()) {
            dictTable.write.insert(records);
        }
        MonotonicTime end = NanoTimeClock.now();

        if (logger.isDebugEnabled()) {
            logger.debug("Written {} new records. Query time: {}", records.size(), end.minus(start).toMillis() / 1e3);
        }
    }

    /**
     * Вынесено отдельным методом в целях упрощения написания юнит-тестов
     */
    LocalDateTime getUtcNow() {
        return ZonedDateTime.now(ZoneOffset.UTC).toLocalDateTime();
    }

}
