package ru.yandex.direct.useractionlog.dict;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.useractionlog.AdGroupId;
import ru.yandex.direct.useractionlog.CampaignId;
import ru.yandex.direct.useractionlog.schema.AdId;
import ru.yandex.direct.useractionlog.schema.ObjectPath;

/**
 * Набор вспомогательных функций для {@link ClickHouseDictRepository}. Осталось от рефакторинга, чтобы не разбухал дифф.
 */
@ParametersAreNonnullByDefault
public class ClickHouseDictUtils {
    // Пусть такая карта обработчиков и выглядит страшнее switch-case, у неё есть преимущество.
    // При добавлении новой категории словарных данных легко забыть прописать её в десятке функций в разных файлах.
    // Метод fullEnumMap проверит, что ничего не забыто, в течение пары секунд после запуска InitDictionaries. В случае
    // со switch-case внутри утилитарных методов, ошибка нашлась бы через несколько часов. Подтверждено на практике.
    private static final Map<DictDataCategory, BiFunction<Long, String, Object>> DESERIALIZER_MAP =
            fullEnumMap(Arrays.asList(
                    Pair.of(DictDataCategory.AD_PATH, (id, value) -> new ObjectPath.AdPath(
                            (ObjectPath.AdGroupPath) ObjectPath.fromPathString(value), new AdId(id))),
                    Pair.of(DictDataCategory.AD_TITLE, ClickHouseDictUtils::writeAsIs),
                    Pair.of(DictDataCategory.ADGROUP_GEO, ClickHouseDictUtils::writeAsIs),
                    Pair.of(DictDataCategory.ADGROUP_NAME, ClickHouseDictUtils::writeAsIs),
                    Pair.of(DictDataCategory.ADGROUP_PATH, (id, value) -> new ObjectPath.AdGroupPath(
                            (ObjectPath.CampaignPath) ObjectPath.fromPathString(value), new AdGroupId(id))),
                    Pair.of(DictDataCategory.CAMPAIGN_DISABLED_IPS, ClickHouseDictUtils::writeAsIs),
                    Pair.of(DictDataCategory.CAMPAIGN_DISABLED_SSP, ClickHouseDictUtils::writeAsIs),
                    Pair.of(DictDataCategory.CAMPAIGN_DONT_SHOW, ClickHouseDictUtils::writeAsIs),
                    Pair.of(DictDataCategory.CAMPAIGN_GEO, ClickHouseDictUtils::writeAsIs),
                    Pair.of(DictDataCategory.CAMPAIGN_MINUS_WORDS, ClickHouseDictUtils::writeAsIs),
                    Pair.of(DictDataCategory.CAMPAIGN_NAME, ClickHouseDictUtils::writeAsIs),
                    Pair.of(DictDataCategory.CAMPAIGN_PATH, (id, value) -> new ObjectPath.CampaignPath(
                            (ObjectPath.ClientPath) ObjectPath.fromPathString(value), new CampaignId(id))),
                    Pair.of(DictDataCategory.CAMPAIGN_STRATEGY_DATA, ClickHouseDictUtils::writeAsIs),
                    Pair.of(DictDataCategory.CAMPAIGN_TIME_TARGET, ClickHouseDictUtils::writeAsIs),
                    Pair.of(DictDataCategory.HIERARCHICAL_MULTIPLIERS_RECORD, ClickHouseDictUtils::writeAsIs),
                    Pair.of(DictDataCategory.RETARGETING_CONDITION_NAME, ClickHouseDictUtils::writeAsIs)));

    private static final Map<DictDataCategory, Function<Object, String>> SERIALIZER_MAP =
            fullEnumMap(Arrays.asList(
                    Pair.of(DictDataCategory.AD_PATH,
                            value -> ((ObjectPath.AdPath) value).getAdGroupPath().toPathString()),
                    Pair.of(DictDataCategory.AD_TITLE, ClickHouseDictUtils::writeAsIs),
                    Pair.of(DictDataCategory.ADGROUP_GEO, ClickHouseDictUtils::writeAsIs),
                    Pair.of(DictDataCategory.ADGROUP_NAME, ClickHouseDictUtils::writeAsIs),
                    Pair.of(DictDataCategory.ADGROUP_PATH,
                            value -> ((ObjectPath.AdGroupPath) value).getCampaignPath().toPathString()),
                    Pair.of(DictDataCategory.CAMPAIGN_DISABLED_IPS, ClickHouseDictUtils::writeAsIs),
                    Pair.of(DictDataCategory.CAMPAIGN_DISABLED_SSP, ClickHouseDictUtils::writeAsIs),
                    Pair.of(DictDataCategory.CAMPAIGN_DONT_SHOW, ClickHouseDictUtils::writeAsIs),
                    Pair.of(DictDataCategory.CAMPAIGN_GEO, ClickHouseDictUtils::writeAsIs),
                    Pair.of(DictDataCategory.CAMPAIGN_MINUS_WORDS, ClickHouseDictUtils::writeAsIs),
                    Pair.of(DictDataCategory.CAMPAIGN_NAME, ClickHouseDictUtils::writeAsIs),
                    Pair.of(DictDataCategory.CAMPAIGN_PATH,
                            value -> ((ObjectPath.CampaignPath) value).getClientPath().toPathString()),
                    Pair.of(DictDataCategory.CAMPAIGN_STRATEGY_DATA, ClickHouseDictUtils::writeAsIs),
                    Pair.of(DictDataCategory.CAMPAIGN_TIME_TARGET, ClickHouseDictUtils::writeAsIs),
                    Pair.of(DictDataCategory.HIERARCHICAL_MULTIPLIERS_RECORD, ClickHouseDictUtils::writeAsIs),
                    Pair.of(DictDataCategory.RETARGETING_CONDITION_NAME, ClickHouseDictUtils::writeAsIs)));

    private ClickHouseDictUtils() {
    }

    @SuppressWarnings("unused")
    private static String writeAsIs(long id, String value) {
        return value;
    }

    @SuppressWarnings("unchecked")
    private static String writeAsIs(Object value) {
        return (String) value;
    }

    /**
     * Проверяет, что есть обработчики для всех значений Enumeration и создаёт ассоциативный массив из enum в
     * обработчики.
     */
    @SuppressWarnings("unchecked")
    private static <K extends Enum<K>, V> Map<K, V> fullEnumMap(List<Map.Entry<K, V>> pairs) {
        ImmutableMap.Builder<K, V> builder = ImmutableMap.builder();
        for (Map.Entry<K, V> pair : pairs) {
            builder.put(pair);
        }
        Map<K, V> result = Maps.immutableEnumMap(builder.build());
        for (K value : (K[]) result.keySet().iterator().next().getClass().getEnumConstants()) {
            if (!result.containsKey(value)) {
                throw new IllegalStateException("Not provided key: " + value);
            }
        }
        return result;
    }

    /**
     * Десериализовать объект из строки, записанный в поле
     * {@link ru.yandex.direct.useractionlog.schema.dict.DictSchema#VALUE} или подобных полях в других таблицах.
     */
    static Object deserializeClickHouse(DictDataCategory type, long id, String value) {
        return DESERIALIZER_MAP.get(type).apply(id, value);
    }

    /**
     * Преобразовать объект в строку для записи в поле
     * {@link ru.yandex.direct.useractionlog.schema.dict.DictSchema#VALUE} или подобные поля в других таблицах.
     */
    static String serializeClickHouse(DictDataCategory type, Object value) {
        return SERIALIZER_MAP.get(type).apply(value);
    }
}
