package ru.yandex.direct.useractionlog.dict;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.useractionlog.schema.ObjectPath;

/**
 * Категория словарных данных.
 */
@ParametersAreNonnullByDefault
public enum DictDataCategory {
    /**
     * В связи с DIRECT-75682 AdPath разрешили быть изменяемым
     */
    AD_PATH(ObjectPath.AdPath.class, true),
    AD_TITLE(String.class, true),
    ADGROUP_GEO(String.class, true),
    ADGROUP_NAME(String.class, true),
    ADGROUP_PATH(ObjectPath.AdGroupPath.class, false),
    CAMPAIGN_DISABLED_IPS(String.class, true),
    CAMPAIGN_DISABLED_SSP(String.class, true),
    CAMPAIGN_DONT_SHOW(String.class, true),
    CAMPAIGN_GEO(String.class, true),
    CAMPAIGN_MINUS_WORDS(String.class, true),
    CAMPAIGN_NAME(String.class, true),
    CAMPAIGN_PATH(ObjectPath.CampaignPath.class, false),
    CAMPAIGN_TIME_TARGET(String.class, true),

    /**
     * Содержимое группы записей из таблицы hierarchical_multipliers и таблиц *_multiplier_values.
     * Ключ - ppc.hierarchical_multipliers.hierarchical_multiplier_id.
     * Значение - два байта, указывающие на версию, затем json-сериализованный {@literal Map<String,String>}.
     */
    HIERARCHICAL_MULTIPLIERS_RECORD(String.class, true),

    /**
     * Название условия ретаргетинга. ppc.retargeting_conditions.condition_name
     */
    RETARGETING_CONDITION_NAME(String.class, true),

    /**
     * json-сериализованный key-value
     */
    CAMPAIGN_STRATEGY_DATA(String.class, true);

    private final Class<?> dataClass;
    private final boolean mutable;

    DictDataCategory(Class<?> dataClass, boolean mutable) {
        this.dataClass = dataClass;
        this.mutable = mutable;
    }

    /**
     * Проверяет, что тип аргумента соответствует категории словарных данных.
     *
     * @param what Словарное значение, готовое для того, чтобы отдать его в генератор пользовательских логов.
     * @return Тот же самый объект, что и на входе.
     * @throws IllegalArgumentException Если тип объекта не соответствует категории.
     */
    public Object validate(Object what) {
        if (!dataClass.isAssignableFrom(what.getClass())) {
            throw new IllegalArgumentException(String.format("Wrong value %s for %s", what, this));
        }
        return what;
    }

    /**
     * Существуют иммутабельные словарные данные, которые однажды создаются и никогда не изменяются. Нет смысла хранить
     * такие данные в бэкендах словарей, которые поддерживают работу с историей изменений данных.
     *
     * @return {@literal true} Если словарные данные этой категории могут меняться со временем.
     */
    public boolean isMutable() {
        return mutable;
    }
}
