package ru.yandex.direct.useractionlog.dict;

import java.util.Arrays;
import java.util.Collection;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * Класс для составления списка запросов в словарь. Передаётся в генератор пользовательских логов чтобы
 * абстрагировать его от знания внутренней кухни словарей.
 * <p>У это класса есть два вида клиентов:
 * <li>пользователи, которые запрашивают данные через require* методы</li>
 * <li>загрузчик, который загружает данные использую переданную коллекцию dictRequests</li>
 */
@ParametersAreNonnullByDefault
public class DictRequestsFiller {
    private final Collection<Collection<DictRequest>> dictRequestCollections;

    /**
     * Конструктор.
     * На вход мы принимаем коллекцию, а не Set, поэтому пользователи и загрузчик сами должны выработать контракт между
     * собой:
     * <li>если пользователи могут вызывать require-методы повторно для одних и тех же аргументов, то загрузчик должен
     * это учитывать и при необходимости де-дублицировать, сделать это проще всего передав Set в качестве dictRequests
     * <li>если пользователи не могут вызывать require-методы повторно для одних и тех же аргументов, то загрузчик</li>
     * может расчитывать на отсутствие дубликатов и в качестве dictRequests можно использовать и List</li>
     *
     * @param dictRequests Набор коллекций, в которые следует добавлять запросы на словарные данные.
     */
    @SafeVarargs
    public DictRequestsFiller(Collection<DictRequest>... dictRequests) {
        dictRequestCollections = Arrays.asList(dictRequests);
    }

    public void require(DictRequest dictRequest) {
        for (Collection<DictRequest> dictRequests : dictRequestCollections) {
            dictRequests.add(dictRequest);
        }
    }

    /**
     * Запросить словарные данные определённой категории.
     */
    public void require(DictDataCategory category, long id) {
        require(new DictRequest(category, id));
    }

    public void requireCampaignPath(long campaignId) {
        require(DictDataCategory.CAMPAIGN_PATH, campaignId);
    }

    /**
     * Запросить название кампании для указанного идентификатора кампании.
     */
    public void requireCampaignNames(long campaignId) {
        require(DictDataCategory.CAMPAIGN_NAME, campaignId);
    }

    public void requireAdGroupPath(long adGroupId) {
        require(DictDataCategory.ADGROUP_PATH, adGroupId);
    }

    /**
     * Запросить название группы для указанного идентификатора группы.
     */
    public void requireAdGroupNames(long adGroupId) {
        require(DictDataCategory.ADGROUP_NAME, adGroupId);
    }

    public void requireAdPath(long adId) {
        require(DictDataCategory.AD_PATH, adId);
    }

    public void requireAdTitle(Long adId) {
        require(DictDataCategory.AD_TITLE, adId);
    }

    public void requireRetargetingConditionName(long retargetingConditionId) {
        require(DictDataCategory.RETARGETING_CONDITION_NAME, retargetingConditionId);
    }
}
