package ru.yandex.direct.useractionlog.dict;

import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.useractionlog.schema.ObjectPath;

/**
 * Класс для организации доступа к полученным ответам на запросы словарных данных. Передаётся в генератор
 * пользовательских логов чтобы абстрагировать его от знания внутренней кухни словарей.
 */
@ParametersAreNonnullByDefault
public class DictResponsesAccessor {
    private final Map<DictRequest, Object> dictData;

    /**
     * @param dictData Все полученные от {@link DictRepository} словарные данные.
     */
    public DictResponsesAccessor(Map<DictRequest, Object> dictData) {
        this.dictData = dictData;
    }

    public Object get(DictRequest dictRequest) {
        if (!dictData.containsKey(dictRequest)) {
            throw new DictDataNotRequested(dictRequest.getCategory(), dictRequest.getId());
        }
        return dictData.get(dictRequest);
    }

    /**
     * Получить словарные данные определённой категории.
     *
     * @throws DictDataNotRequested Если словарные данные не были запрошены с помощью {@link DictRequestsFiller}.
     *                              Свидетельствует об ошибке в генераторе логов.
     */
    public Object get(DictDataCategory category, long id) {
        return get(new DictRequest(category, id));
    }

    /**
     * Получить {@link ObjectPath.CampaignPath} по идентификатору кампании.
     *
     * @param campaignId Идентификатор кампании.
     * @throws NullPointerException Если идентификатор кампании не был указан в
     *                              {@link DictRequestsFiller#requireCampaignPath(long)}.
     *                              Свидетельствует об ошибке в генераторе логов.
     */
    public ObjectPath.CampaignPath getCampaignPath(long campaignId) {
        return (ObjectPath.CampaignPath) get(DictDataCategory.CAMPAIGN_PATH, campaignId);
    }

    /**
     * Получить название кампании по идентификатору кампании.
     *
     * @param campaignId Идентификатор кампании.
     * @throws NullPointerException Если идентификатор кампании не был указан в
     *                              {@link DictRequestsFiller#requireCampaignNames(long)}.
     *                              Свидетельствует об ошибке в генераторе логов.
     */
    public String getCampaignName(long campaignId) {
        return (String) get(DictDataCategory.CAMPAIGN_NAME, campaignId);
    }

    public ObjectPath.AdGroupPath getAdGroupPath(long adGroupId) {
        return (ObjectPath.AdGroupPath) get(DictDataCategory.ADGROUP_PATH, adGroupId);
    }

    /**
     * Получить название группы по идентификатору группы.
     *
     * @param adGroupId Идентификатор группы.
     * @throws NullPointerException Если идентификатор группы не был указан в
     *                              {@link DictRequestsFiller#requireAdGroupNames(long)}.
     *                              Свидетельствует об ошибке в генераторе логов.
     */
    public String getAdGroupName(long adGroupId) {
        return (String) get(DictDataCategory.ADGROUP_NAME, adGroupId);
    }

    public ObjectPath.AdPath getAdPath(long adId) {
        return (ObjectPath.AdPath) get(DictDataCategory.AD_PATH, adId);
    }

    public String getAdTitle(long adId) {
        return (String) get(DictDataCategory.AD_TITLE, adId);
    }

    public String getRetargetingConditionName(long retargetingConditionId) {
        return (String) get(DictDataCategory.RETARGETING_CONDITION_NAME, retargetingConditionId);
    }
}
