package ru.yandex.direct.useractionlog.model;

import java.util.LinkedHashMap;
import java.util.Objects;

import javax.annotation.Nonnull;

import org.apache.commons.lang3.tuple.Triple;

import ru.yandex.direct.binlogclickhouse.schema.FieldValueList;
import ru.yandex.direct.useractionlog.AdGroupId;
import ru.yandex.direct.useractionlog.CampaignId;
import ru.yandex.direct.useractionlog.schema.AdId;

import static ru.yandex.direct.dbschema.ppc.Ppc.PPC;

public class AdRowModel extends RowModel {
    private static final String VERSION = "0";

    private static final String CAMPAIGN_ID_NAME = PPC.BANNERS.CID.getName();
    private static final String AD_GROUP_ID_NAME = PPC.BANNERS.PID.getName();
    private static final String AD_ID_NAME = PPC.BANNERS.BID.getName();
    public static final AdGroupId ZERO_AD_GROUP_ID = new AdGroupId(0);

    public AdRowModel(LinkedHashMap<String, String> map) {
        super(map);
    }

    public AdRowModel() {
    }

    public AdRowModel(FieldValueList list) {
        super(list);

    }

    @Override
    public String getVersion() {
        return VERSION;
    }

    public CampaignId getCampaignId() {
        String value = map.get(CAMPAIGN_ID_NAME);
        if (value == null) {
            return null;
        }
        return new CampaignId(Long.parseLong(value));
    }

    public AdGroupId getAdGroupId() {
        String value = map.get(AD_GROUP_ID_NAME);
        if (value == null) {
            return null;
        }
        return new AdGroupId(Long.parseLong(value));
    }

    public AdId getAdId() {
        String value = map.get(AD_ID_NAME);
        if (value == null) {
            return null;
        }
        return new AdId(Long.parseLong(value));
    }

    /**
     * {@inheritDoc}
     * <p>
     * В медиапланах есть код, который сначала создаёт баннеры с pid=0, а потом меняет pid на другой.
     * <p>
     * В DIRECT-75682 решили, что такие изменения разрешены. Очевидная проблема - в базе один и тот же баннер
     * будет иметь два разных ObjectPath, пользователь сможет увидеть всю историю баннера только если
     * задаст запрос для нового pid и для pid=0.
     * <p>
     * С другой стороны, бывают запросы в тестовых приложениях, которые меняют pid на 0
     * direct/qa/api5-ads/src/main/java/ru/yandex/autotests/direct/api/ads/get/AdWithoutAdGroupId.java
     */
    @Override
    boolean isValidObjectPathChange(RowModel other) {
        if (other instanceof AdRowModel) {
            Triple<CampaignId, AdGroupId, AdId> oldParts = getObjectPathParts();
            Triple<CampaignId, AdGroupId, AdId> newParts = ((AdRowModel) other).getObjectPathParts();
            if (Objects.equals(oldParts.getMiddle(), ZERO_AD_GROUP_ID)
                    || Objects.equals(newParts.getMiddle(), ZERO_AD_GROUP_ID)
            ) {
                oldParts = Triple.of(
                        oldParts.getLeft(),
                        newParts.getMiddle(),
                        oldParts.getRight());
            }
            return oldParts.equals(newParts);
        } else {
            return false;
        }
    }

    @Override
    @Nonnull
    Triple<CampaignId, AdGroupId, AdId> getObjectPathParts() {
        CampaignId campaignId = getCampaignId();
        AdGroupId adGroupId = getAdGroupId();
        AdId adId = getAdId();
        return Triple.of(campaignId, adGroupId, adId);
    }
}
