package ru.yandex.direct.useractionlog.model;

import java.util.LinkedHashMap;

import javax.annotation.Nonnull;

import com.google.common.base.Preconditions;
import org.apache.commons.lang3.tuple.Triple;

import ru.yandex.direct.binlogclickhouse.schema.FieldValueList;
import ru.yandex.direct.useractionlog.AdGroupId;
import ru.yandex.direct.useractionlog.CampaignId;
import ru.yandex.direct.useractionlog.ClientId;
import ru.yandex.direct.useractionlog.schema.HierarchicalMultiplierId;
import ru.yandex.direct.useractionlog.schema.ObjectPath;

import static ru.yandex.direct.dbschema.ppc.Ppc.PPC;

/**
 * С моделью можно работать в двух режимах:
 * 1) как с {@link HierarchicalMultipliersData.Root}. Для этого есть конструктор
 * и набор обёрток над методами.
 * 2) как с обёткой над {@link LinkedHashMap}. Можно либо создать модель в таком
 * виде, либо конвертировать с помощью {@link HierarchicalMultiplierRowModel#convertToMap()}.
 * После этого переменная {@link HierarchicalMultiplierRowModel#root} считается невалидной.
 */
public class HierarchicalMultiplierRowModel extends RowModel {
    private static final String CLIENT_ID_NAME = PPC.CLIENTS.CLIENT_ID.getName();
    private static final String CAMPAIGN_ID_NAME = PPC.HIERARCHICAL_MULTIPLIERS.CID.getName();
    private static final String AD_GROUP_ID_NAME = PPC.HIERARCHICAL_MULTIPLIERS.PID.getName();

    private static final String VERSION = HierarchicalMultipliersData.getCurrentVersion();
    private HierarchicalMultipliersData.Root root;
    private HierarchicalMultiplierId hierarchicalMultiplierId;
    private boolean isInMapState;

    public HierarchicalMultiplierRowModel() {
        super();
        isInMapState = true;
    }

    public HierarchicalMultiplierRowModel(LinkedHashMap<String, String> map) {
        super(map);
        isInMapState = true;
    }

    public HierarchicalMultiplierRowModel(FieldValueList list) {
        super(list);
        isInMapState = true;
    }

    public HierarchicalMultiplierRowModel(HierarchicalMultipliersData.Root root,
                                          HierarchicalMultiplierId hierarchicalMultiplierId) {
        super();
        this.root = root;
        this.hierarchicalMultiplierId = hierarchicalMultiplierId;
        isInMapState = false;
    }

    public void checkNotMapState() {
        Preconditions.checkState(!isInMapState, "Model in map state");
    }

    public void checkMapState() {
        Preconditions.checkState(isInMapState, "Model not in map state");
    }

    @Override
    public String getVersion() {
        return VERSION;
    }

    public ClientId getClientId() {
        if (!isInMapState) {
            ObjectPath objectPath = root.getPath();
            switch (objectPath.getPathElementType()) {
                case CAMPAIGN:
                    return ((ObjectPath.CampaignPath) objectPath).getClientId();
                case ADGROUP:
                    return ((ObjectPath.AdGroupPath) objectPath).getClientId();
                default:
                    throw new IllegalStateException(
                            "unknown root ObjectPath " + objectPath.getPathElementType().toString());
            }
        } else {
            String value = map.get(CLIENT_ID_NAME);
            if (value == null) {
                return null;
            }
            return new ClientId(Long.parseLong(value));
        }
    }

    public CampaignId getCampaignId() {
        if (!isInMapState) {
            ObjectPath objectPath = root.getPath();
            switch (objectPath.getPathElementType()) {
                case CAMPAIGN:
                    return ((ObjectPath.CampaignPath) objectPath).getId();
                case ADGROUP:
                    return ((ObjectPath.AdGroupPath) objectPath).getCampaignId();
                default:
                    throw new IllegalStateException(
                            "unknown root ObjectPath " + objectPath.getPathElementType().toString());
            }
        } else {
            String value = map.get(CAMPAIGN_ID_NAME);
            if (value == null) {
                return null;
            }
            return new CampaignId(Long.parseLong(value));
        }
    }

    public AdGroupId getAdGroupId() {
        if (!isInMapState) {
            ObjectPath objectPath = root.getPath();
            switch (objectPath.getPathElementType()) {
                case CAMPAIGN:
                    return null;
                case ADGROUP:
                    return ((ObjectPath.AdGroupPath) objectPath).getId();
                default:
                    throw new IllegalStateException(
                            "unknown root ObjectPath " + objectPath.getPathElementType().toString());
            }
        } else {
            String value = map.get(AD_GROUP_ID_NAME);
            if (value == null) {
                return null;
            }
            return new AdGroupId(Long.parseLong(value));
        }
    }

    @Override
    @Nonnull
    Triple<ClientId, CampaignId, AdGroupId> getObjectPathParts() {
        ClientId clientId = getClientId();
        CampaignId campaignId = getCampaignId();
        AdGroupId adGroupId = getAdGroupId();
        return Triple.of(clientId, campaignId, adGroupId);
    }


    /* Функции доступные после конвертации в map */
    @Override
    public LinkedHashMap<String, String> getMap() {
        checkMapState();
        return map;
    }

    @Override
    public FieldValueList toFieldValueList() {
        checkMapState();
        return super.toFieldValueList();
    }

    public HierarchicalMultiplierId getHierachicalMultiplierId() {
        if (isInMapState) {
            String value = map.get(PPC.HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.getName());
            if (value == null) {
                return null;
            }
            return new HierarchicalMultiplierId(Long.parseLong(value));
        } else {
            return hierarchicalMultiplierId;
        }
    }

    /* Функции доступные до конвертации в map*/
    public HierarchicalMultipliersData.Root getRoot() {
        checkNotMapState();
        return root;
    }

    public HierarchicalMultiplierRowModel convertToMap() {
        checkNotMapState();
        map = HierarchicalMultipliersData.toFieldValueList(root, hierarchicalMultiplierId.toLong()).toMap();
        isInMapState = true;
        return this;
    }
}
