package ru.yandex.direct.useractionlog.model;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.NotThreadSafe;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.binlog.reader.MySQLSimpleRowIndexed;
import ru.yandex.direct.binlogclickhouse.schema.FieldValue;
import ru.yandex.direct.binlogclickhouse.schema.FieldValueList;
import ru.yandex.direct.dbschema.ppc.enums.HierarchicalMultipliersType;
import ru.yandex.direct.useractionlog.schema.ObjectPath;
import ru.yandex.direct.utils.Checked;

import static ru.yandex.direct.dbschema.ppc.Ppc.PPC;

/**
 * Структуры данных для чтения, хранения и записи группы корректировок ставок.
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
@NotThreadSafe
@ParametersAreNonnullByDefault
public abstract class HierarchicalMultipliersData<T extends HierarchicalMultipliersData> {
    public static final Set<String> TYPES_WITHOUT_RELATED_TABLES = ImmutableSet.of(
            HierarchicalMultipliersType.mobile_multiplier.getLiteral(),
            HierarchicalMultipliersType.video_multiplier.getLiteral(),
            HierarchicalMultipliersType.performance_tgo_multiplier.getLiteral());

    private static final ObjectMapper MAPPER = new ObjectMapper();
    private static final String CURRENT_VERSION = "01";
    private static final String S_AGE = "age";
    private static final String S_GENDER = "gender";
    private static final String S_IS_ENABLED = "is_enabled";
    private static final String S_IS_HIDDEN = "is_hidden";
    private static final String S_LAST_CHANGE = "last_change";
    private static final String S_MULTIPLIER_PCT = "multiplier_pct";
    private static final String S_PATH = "path";
    private static final String S_REGION_ID = "region_id";
    private static final String S_RELATED_IDS = "related_ids";
    private static final String S_RET_COND_ID = "ret_cond_id";
    private static final String S_RET_COND_NAME = "ret_cond_name";
    private static final String S_TYPE = "type";
    private static final String S_VERSION = RowModel.getVersionField();
    // Поле было переименовано в DIRECT-73983, но остались записи со старым названием
    private static final String S_OLD_VERSION = "version";

    public static String getCurrentVersion() {
        return CURRENT_VERSION;
    }

    /**
     * Десериализация из строки, которая была получена из словаря
     */
    public static HierarchicalMultipliersData.Root fromDictValue(String raw) {
        String version = raw.substring(0, 2);
        if (!version.equals(CURRENT_VERSION)) {
            throw new IllegalStateException("Can't deserialize value with version " + version);
        }
        try {
            @SuppressWarnings("unchecked")
            Map<String, String> deserialized = Checked.get(() -> MAPPER.readValue(raw.substring(2), Map.class));
            deserialized.put(S_VERSION, version);
            return new HierarchicalMultipliersData.Root()
                    .withPath(ObjectPath.fromPathString(deserialized.get(S_PATH)))
                    .updateFromMap(deserialized)
                    .checkNonNull("Failed to deserialize dict value");
        } catch (ClassCastException e) {
            throw new IllegalStateException("Not all keys and values are strings: " + raw, e);
        }
    }

    /**
     * Представление всей группы корректировок ставок в виде кортежа.
     *
     * @param hierarchicalMultipliersId Идентификатор группы записей
     */
    public static FieldValueList toFieldValueList(HierarchicalMultipliersData.Root data,
                                                  long hierarchicalMultipliersId) {
        data.checkNonNull("");
        List<Map.Entry<String, String>> entries = data.toNameValueList("");
        entries.add(Pair.of("hierarchical_multiplier_id", String.valueOf(hierarchicalMultipliersId)));
        entries.add(Pair.of(S_VERSION, CURRENT_VERSION));
        return new FieldValueList(entries.stream()
                .sorted(Comparator.comparing(Map.Entry::getKey))
                .map(pair -> new FieldValue<>(pair.getKey(), pair.getValue()))
                .collect(Collectors.toList()));
    }

    /**
     * Сериализация в строку, которая будет записана в словарь
     */
    public static String toDictValue(HierarchicalMultipliersData.Root data) {
        data.checkNonNull("");
        Map<String, String> map = new HashMap<>();
        for (Map.Entry<String, String> stringStringEntry : data.toNameValueList("")) {
            map.put(stringStringEntry.getKey(), stringStringEntry.getValue());
        }
        map.put("path", data.path.toPathString());
        return Checked.get(() -> CURRENT_VERSION + MAPPER.writeValueAsString(map));
    }

    private static <V> void applyIfPresent(Map<String, V> map, String key, Function<V, ?> consumer) {
        V obj = map.get(key);
        if (obj != null) {
            consumer.apply(obj);
        }
    }

    /**
     * Записать все данные, которые есть в кортеже.
     *
     * @param rowData Кортеж, который был получен из базы данных или из бинлога.
     * @return this
     */
    public abstract T update(MySQLSimpleRowIndexed rowData);

    /**
     * Проверка, что все поля структур заполнены.
     *
     * @param msg Сообщение об ошибке, если хотя бы одно поле не заполнено.
     * @return this
     */
    public abstract T checkNonNull(String msg);

    /**
     * Представление всей группы ставок в виде набора пар ключ-значение.
     *
     * @param nameSuffix Строка, которая будет добавлена к каждому ключу.
     * @return Набор пар ключ-значение.
     */
    public abstract List<Map.Entry<String, String>> toNameValueList(String nameSuffix);

    /**
     * Записать все данные, которые есть в ассоциативном массиве.
     * <p>
     * Для любого t типа T и любой строки n должно выполняться условие: t содержит те же данные, что и
     * {@literal new T().updateFromMap(new HashMap<>(t.toNameValueList(n), n))}.
     *
     * @param deserialized Ассоциативный массив с исходными данными.
     * @param nameSuffix   Строчка, которую нужно убрать с конца каждого ключа. Если ключ не заканчивается на эту
     *                     строчку, то такая пара ключ-значение игнорируется.
     * @return this
     */
    abstract T updateFromMap(Map<String, String> deserialized, String nameSuffix);

    /**
     * Вся группа корректировок ставок.
     */
    public static class Root extends HierarchicalMultipliersData<Root> {
        private ObjectPath path;
        private String lastChange;
        private String type;
        private String multiplierPct;
        private String isEnabled;
        private Map<Long, Integer> databaseIdToInternalId = new HashMap<>();
        private SortedMap<Integer, HierarchicalMultipliersData> internalIdToRelatedData = new TreeMap<>();

        /**
         * Путь до группы записей. Может быть {@link ObjectPath.CampaignPath} или {@link ObjectPath.AdGroupPath}.
         */
        public ObjectPath getPath() {
            return path;
        }

        public HierarchicalMultipliersData.Root withPath(ObjectPath path) {
            this.path = path;
            return this;
        }

        /**
         * Дата последнего изменения во всей группе. Содержит в себе содержимое из полей last_updated таблиц
         * hierarchical_multipliers и *_multiplier_values.
         * Изменяется только при добавлении и изменении корректировки, не меняется при удалении корректировки.
         */
        public String getLastChange() {
            return lastChange;
        }

        public HierarchicalMultipliersData.Root withLastChange(String lastChange) {
            this.lastChange = lastChange;
            return this;
        }

        /**
         * ppc.hierarchical_multipliers.type
         */
        public String getType() {
            return type;
        }

        public HierarchicalMultipliersData.Root withType(String type) {
            this.type = type;
            return this;
        }

        /**
         * ppc.hierarchical_multipliers.multiplier_pct
         * <p>
         * Единый множитель корректировки ставки для типов, которые не могут содержать разные корректировки для разных
         * условий. Для остальных типов - пустая строка.
         * К таким типам относятся mobile_multiplier, video_multiplier и performance_tgo_multiplier.
         */
        public String getMultiplierPct() {
            return multiplierPct;
        }

        public HierarchicalMultipliersData.Root withMultiplierPct(String multiplierPct) {
            this.multiplierPct = multiplierPct;
            return this;
        }

        /**
         * ppc.hierarchical_multipliers.is_enabled
         */
        public String getIsEnabled() {
            return isEnabled;
        }

        public HierarchicalMultipliersData.Root withIsEnabled(String isEnabled) {
            this.isEnabled = isEnabled;
            return this;
        }

        /**
         * Группа корректировок не содержит не содержит записей (или это mobile_multiplier/video_multiplier/etc)
         */
        public boolean relatedIsEmpty() {
            return internalIdToRelatedData.isEmpty();
        }

        /**
         * Количество корректировок в группе
         */
        public int relatedSize() {
            return internalIdToRelatedData.size();
        }

        /**
         * Группа содержит корректировку из какой-либо таблицы *_multiplier_values
         *
         * @param databaseId Значение PRIMARY KEY из таблицы *_multiplier_values
         */
        public boolean relatedContainsKey(long databaseId) {
            return databaseIdToInternalId.containsKey(databaseId);
        }

        public Collection<Long> getAllRelatedDatabaseIds() {
            return databaseIdToInternalId.keySet();
        }

        /**
         * Получить структуру с данными об отдельной корректировке. Работает как Map.get.
         *
         * @param databaseId Значение PRIMARY KEY из таблицы *_multiplier_values
         */
        public HierarchicalMultipliersData getRelated(long databaseId) {
            Integer internalId = databaseIdToInternalId.get(databaseId);
            if (internalId != null) {
                return Objects.requireNonNull(internalIdToRelatedData.get(internalId),
                        "Bug. One map has entry but no appropriate entry in another map.");
            } else {
                return null;
            }
        }

        /**
         * Записать структуру с данными об отдельной корректировке. Работает как Map.put.
         *
         * @param databaseId Значение PRIMARY KEY из таблицы *_multiplier_values
         */
        public HierarchicalMultipliersData putRelated(long databaseId, HierarchicalMultipliersData data) {
            int internalId = databaseIdToInternalId.computeIfAbsent(databaseId, k -> {
                int candidate = 0;
                for (Integer existingInternalId : internalIdToRelatedData.keySet()) {
                    if (candidate < existingInternalId) {
                        break;
                    } else {
                        candidate = existingInternalId + 1;
                    }
                }
                return candidate;
            });
            return internalIdToRelatedData.put(internalId, data);
        }

        /**
         * Удалить структуру с данными об отдельной корректировке. Работает как Map.remove.
         *
         * @param databaseId Значение PRIMARY KEY из таблицы *_multiplier_values
         */
        public HierarchicalMultipliersData removeRelated(long databaseId) {
            Integer internalId = databaseIdToInternalId.remove(databaseId);
            if (internalId != null) {
                return Objects.requireNonNull(internalIdToRelatedData.remove(internalId),
                        "Bug. One map has entry but no appropriate entry in another map.");
            } else {
                return null;
            }
        }

        @Override
        public HierarchicalMultipliersData.Root update(MySQLSimpleRowIndexed data) {
            return this
                    .withLastChange(
                            data.getByName(PPC.HIERARCHICAL_MULTIPLIERS.LAST_CHANGE.getName()).getValueAsString())
                    .withType(
                            data.getByName(PPC.HIERARCHICAL_MULTIPLIERS.TYPE.getName()).getValueAsString())
                    .withMultiplierPct(Optional.ofNullable(
                            data.getByName(
                                    PPC.HIERARCHICAL_MULTIPLIERS.MULTIPLIER_PCT.getName()).getValueAsString())
                            .orElse(""))
                    .withIsEnabled(
                            data.getByName(PPC.HIERARCHICAL_MULTIPLIERS.IS_ENABLED.getName()).getValueAsString());
        }

        /**
         * Записать все данные, которые есть в ассоциативном массиве.
         * <p>
         * Для любого t типа T и любой строки n должно выполняться условие: t содержит те же данные, что и
         * {@literal new T().updateFromMap(new HashMap<>(t.toNameValueList(n), n))}.
         *
         * @param deserialized Ассоциативный массив с исходными данными.
         * @return this
         */
        public HierarchicalMultipliersData.Root updateFromMap(Map<String, String> deserialized) {
            return updateFromMap(deserialized, "");
        }

        /**
         * Записать все данные, которые есть в ассоциативном массиве.
         * <p>
         * Для любого t типа T и любой строки n должно выполняться условие: t содержит те же данные, что и
         * {@literal new T().updateFromMap(new HashMap<>(t.toNameValueList(n)), n)}.
         * <p>
         * Является аналогом <code>object.updateFromMap(oldDeserialized).updateFromMap(newDeserialized)</code>,
         * но ряд иммутабельных полей будут взяты из oldDeserialized, если они отсутствуют в newDeserialized.
         * Предназначается для десериализации из UPDATE-записи ActionLogRecord, в которых из-за дедупликации удаляется
         * метаинформация из after.
         */
        public HierarchicalMultipliersData.Root updateFromMaps(Map<String, String> oldDeserialized,
                                                               Map<String, String> newDeserialized) {
            updateFromMap(oldDeserialized);
            oldDeserialized.forEach(newDeserialized::putIfAbsent);
            updateFromMap(newDeserialized);
            return this;
        }

        @Override
        @SuppressWarnings("unchecked")
        public List<Map.Entry<String, String>> toNameValueList(String nameSuffix) {
            if (!nameSuffix.isEmpty()) {
                throw new UnsupportedOperationException();
            }
            List<Map.Entry<String, String>> namesAndValues = new ArrayList<>();

            namesAndValues.add(Pair.of(S_LAST_CHANGE, lastChange));
            namesAndValues.add(Pair.of(S_IS_ENABLED, isEnabled));
            namesAndValues.add(Pair.of(S_TYPE, type));
            if (TYPES_WITHOUT_RELATED_TABLES.contains(type)) {
                namesAndValues.add(Pair.of(S_MULTIPLIER_PCT, multiplierPct));
            }
            namesAndValues.add(Pair.of(S_RELATED_IDS, databaseIdToInternalId.entrySet().stream()
                    .sorted(Comparator.comparing(Map.Entry::getValue))
                    .map(e -> "" + e.getValue() + "-" + e.getKey())
                    .collect(Collectors.joining(","))));
            for (Map.Entry<Integer, HierarchicalMultipliersData> entry : internalIdToRelatedData.entrySet()) {
                HierarchicalMultipliersData value = entry.getValue();
                List<Pair<String, String>> nameValueList = value.toNameValueList("_" + entry.getKey().toString());
                namesAndValues.addAll(nameValueList);
            }
            namesAndValues.sort(Comparator.comparing(Map.Entry::getKey));
            return namesAndValues;
        }


        @Override
        public HierarchicalMultipliersData.Root checkNonNull(String msg) {
            Objects.requireNonNull(path, msg);
            Objects.requireNonNull(lastChange, msg);
            Objects.requireNonNull(type, msg);
            Objects.requireNonNull(multiplierPct, msg);
            Objects.requireNonNull(isEnabled, msg);
            internalIdToRelatedData.forEach((k, v) -> v.checkNonNull(msg + ": " + k));
            return this;
        }

        @Override
        public Root updateFromMap(Map<String, String> deserialized, String nameSuffix) {
            if (!nameSuffix.isEmpty()) {
                throw new UnsupportedOperationException();
            }
            String version = deserialized.containsKey(S_VERSION)
                    ? deserialized.get(S_VERSION)
                    : deserialized.get(S_OLD_VERSION);
            if (!version.equals(CURRENT_VERSION)) {
                throw new IllegalStateException("Can't deserialize value with version " + version);
            }
            applyIfPresent(deserialized, S_IS_ENABLED, this::withIsEnabled);
            applyIfPresent(deserialized, S_LAST_CHANGE, this::withLastChange);
            applyIfPresent(deserialized, S_TYPE, this::withType);
            if (TYPES_WITHOUT_RELATED_TABLES.contains(type)) {
                applyIfPresent(deserialized, S_MULTIPLIER_PCT, this::withMultiplierPct);
                return this;
            } else {
                this.withMultiplierPct("");
            }

            Pattern internalIdToDatabaseIdPattern = Pattern.compile("^([0-9]+)-([0-9]+)$");
            Set<Long> keepTheseDatabaseIds = new HashSet<>();
            Set<Integer> keepTheseInternalIds = new HashSet<>();
            if (!deserialized.get(S_RELATED_IDS).isEmpty()) {
                for (String serializedEntry : deserialized.get(S_RELATED_IDS).split(",")) {
                    Matcher matcher = internalIdToDatabaseIdPattern.matcher(serializedEntry);
                    if (!matcher.matches()) {
                        throw new IllegalStateException(
                                "Can't parse \"" + serializedEntry + "\" as numbers separated by dash.");
                    }
                    long databaseId = Long.parseLong(matcher.group(2));
                    int internalId = Integer.parseInt(matcher.group(1));
                    databaseIdToInternalId.put(databaseId, internalId);
                    keepTheseDatabaseIds.add(databaseId);
                    keepTheseInternalIds.add(internalId);
                }
            }
            databaseIdToInternalId.keySet().removeIf(databaseId -> !keepTheseDatabaseIds.contains(databaseId));
            internalIdToRelatedData.keySet().removeIf(internalId -> !keepTheseInternalIds.contains(internalId));

            Supplier<HierarchicalMultipliersData> relatedMaker;
            if (type.equals(HierarchicalMultipliersType.demography_multiplier.getLiteral())) {
                relatedMaker = Demography::new;
            } else if (type.equals(HierarchicalMultipliersType.geo_multiplier.getLiteral())) {
                relatedMaker = Geo::new;
            } else if (type.equals(HierarchicalMultipliersType.retargeting_multiplier.getLiteral())) {
                relatedMaker = Retargeting::new;
            } else {
                // DIRECT-74296 здесь должно быть unchecked exception
                // такую ситуацию надо ловить и игнорировать запись/чтение
                throw new IllegalStateException("Unexpected type " + type);
            }
            Pattern keySuffixPattern = Pattern.compile("^.*(_([0-9]+))$");
            for (String key : deserialized.keySet()) {
                Matcher matcher = keySuffixPattern.matcher(key);
                if (matcher.matches()) {
                    String suffix = matcher.group(1);
                    int id = Integer.parseInt(matcher.group(2));
                    internalIdToRelatedData.computeIfAbsent(id, k -> relatedMaker.get())
                            .updateFromMap(deserialized, suffix);
                }
            }
            return this;
        }
    }

    /**
     * Данные о корректировке ставок для отдельного пола и возраста.
     */
    public static class Demography extends HierarchicalMultipliersData<Demography> {
        private String age;
        private String gender;
        private String multiplierPct;

        /**
         * ppc.demography_multiplier_values.age
         */
        public String getAge() {
            return age;
        }

        public Demography withAge(String age) {
            this.age = age;
            return this;
        }

        /**
         * ppc.demography_multiplier_values.gender
         */
        public String getGender() {
            return gender;
        }

        public Demography withGender(String gender) {
            this.gender = gender;
            return this;
        }

        /**
         * ppc.demography_multiplier_values.multiplier_pct
         */
        public String getMultiplierPct() {
            return multiplierPct;
        }

        public Demography withMultiplierPct(String multiplierPct) {
            this.multiplierPct = multiplierPct;
            return this;
        }

        @Override
        public HierarchicalMultipliersData.Demography update(MySQLSimpleRowIndexed rowData) {
            return this
                    .withAge(Optional.ofNullable(
                            rowData.getByName(PPC.DEMOGRAPHY_MULTIPLIER_VALUES.AGE.getName())
                                    .getValueAsString())
                            .orElse(""))
                    .withGender(Optional.ofNullable(
                            rowData.getByName(PPC.DEMOGRAPHY_MULTIPLIER_VALUES.GENDER.getName())
                                    .getValueAsString())
                            .orElse(""))
                    .withMultiplierPct(rowData.getByName(PPC.DEMOGRAPHY_MULTIPLIER_VALUES.MULTIPLIER_PCT.getName())
                            .getValueAsString());
        }

        @Override
        public Demography checkNonNull(String msg) {
            Objects.requireNonNull(age, msg);
            Objects.requireNonNull(gender, msg);
            Objects.requireNonNull(multiplierPct, msg);
            return this;
        }

        @Override
        public List<Map.Entry<String, String>> toNameValueList(String nameSuffix) {
            return Arrays.asList(
                    Pair.of(S_AGE + nameSuffix, age),
                    Pair.of(S_GENDER + nameSuffix, gender),
                    Pair.of(S_MULTIPLIER_PCT + nameSuffix, multiplierPct));
        }

        @Override
        public Demography updateFromMap(Map<String, String> deserialized, String nameSuffix) {
            applyIfPresent(deserialized, S_AGE + nameSuffix, this::withAge);
            applyIfPresent(deserialized, S_GENDER + nameSuffix, this::withGender);
            applyIfPresent(deserialized, S_MULTIPLIER_PCT + nameSuffix, this::withMultiplierPct);
            return this;
        }
    }

    /**
     * Данные о корректировке для отдельного региона.
     */
    public static class Geo extends HierarchicalMultipliersData<Geo> {
        private String isHidden;
        private String multiplierPct;
        private String regionId;

        /**
         * ppc.geo_multiplier_values.is_hidden
         */
        public String getIsHidden() {
            return isHidden;
        }

        public Geo withIsHidden(String isHidden) {
            this.isHidden = isHidden;
            return this;
        }

        /**
         * ppc.geo_multiplier_values.multiplier_pct
         */
        public String getMultiplierPct() {
            return multiplierPct;
        }

        public Geo withMultiplierPct(String multiplierPct) {
            this.multiplierPct = multiplierPct;
            return this;
        }

        /**
         * ppc.geo_multiplier_values.region_id
         */
        public String getRegionId() {
            return regionId;
        }

        public Geo withRegionId(String regionId) {
            this.regionId = regionId;
            return this;
        }

        @Override
        public HierarchicalMultipliersData.Geo update(MySQLSimpleRowIndexed rowData) {
            return this
                    .withIsHidden(rowData.getByName(PPC.GEO_MULTIPLIER_VALUES.IS_HIDDEN.getName())
                            .getValueAsString())
                    .withMultiplierPct(rowData.getByName(PPC.GEO_MULTIPLIER_VALUES.MULTIPLIER_PCT.getName())
                            .getValueAsString())
                    .withRegionId(rowData.getByName(PPC.GEO_MULTIPLIER_VALUES.REGION_ID.getName())
                            .getValueAsString());
        }

        @Override
        public HierarchicalMultipliersData.Geo checkNonNull(String msg) {
            Objects.requireNonNull(isHidden, msg);
            Objects.requireNonNull(multiplierPct, msg);
            Objects.requireNonNull(regionId, msg);
            return this;
        }

        @Override
        public List<Map.Entry<String, String>> toNameValueList(String nameSuffix) {
            return Arrays.asList(
                    Pair.of(S_IS_HIDDEN + nameSuffix, isHidden),
                    Pair.of(S_MULTIPLIER_PCT + nameSuffix, multiplierPct),
                    Pair.of(S_REGION_ID + nameSuffix, regionId));
        }

        @Override
        public Geo updateFromMap(Map<String, String> deserialized, String nameSuffix) {
            applyIfPresent(deserialized, S_IS_HIDDEN + nameSuffix, this::withIsHidden);
            applyIfPresent(deserialized, S_MULTIPLIER_PCT + nameSuffix, this::withMultiplierPct);
            applyIfPresent(deserialized, S_REGION_ID + nameSuffix, this::withRegionId);
            return this;
        }
    }

    /**
     * Данные о корректировке ставки для отдельного условия показов.
     */
    public static class Retargeting extends HierarchicalMultipliersData<Retargeting> {
        private String retCondId;
        private String retCondName;
        private String multiplierPct;

        /**
         * ppc.retargeting_multiplier_values.ret_cond_id
         */
        public String getRetCondId() {
            return retCondId;
        }

        public Retargeting withRetCondId(String retCondId) {
            this.retCondId = retCondId;
            return this;
        }

        /**
         * ppc.retargeting_conditions.condition_name для соответсвтующего ppc.retargeting_multiplier_values.ret_cond_id
         */
        public String getRetCondName() {
            return retCondName;
        }

        public Retargeting withRetCondName(String retCondName) {
            this.retCondName = retCondName;
            return this;
        }

        /**
         * ppc.retargeting_multiplier_values.multiplier_pct
         */
        public String getMultiplierPct() {
            return multiplierPct;
        }

        public Retargeting withMultiplierPct(String multiplierPct) {
            this.multiplierPct = multiplierPct;
            return this;
        }

        @Override
        public Retargeting update(MySQLSimpleRowIndexed rowData) {
            return this
                    .withRetCondId(rowData.getByName(PPC.RETARGETING_MULTIPLIER_VALUES.RET_COND_ID.getName())
                            .getValueAsString())
                    .withMultiplierPct(rowData.getByName(PPC.RETARGETING_MULTIPLIER_VALUES.MULTIPLIER_PCT.getName())
                            .getValueAsString());
        }

        @Override
        public Retargeting checkNonNull(String msg) {
            Objects.requireNonNull(retCondId, msg);
            Objects.requireNonNull(retCondName, msg);
            Objects.requireNonNull(multiplierPct, msg);
            return this;
        }

        @Override
        public List<Map.Entry<String, String>> toNameValueList(String nameSuffix) {
            return Arrays.asList(
                    Pair.of(S_RET_COND_ID + nameSuffix, retCondId),
                    Pair.of(S_RET_COND_NAME + nameSuffix, retCondName),
                    Pair.of(S_MULTIPLIER_PCT + nameSuffix, multiplierPct));
        }

        @Override
        public Retargeting updateFromMap(Map<String, String> deserialized, String nameSuffix) {
            applyIfPresent(deserialized, S_RET_COND_ID + nameSuffix, this::withRetCondId);
            applyIfPresent(deserialized, S_RET_COND_NAME + nameSuffix, this::withRetCondName);
            applyIfPresent(deserialized, S_MULTIPLIER_PCT + nameSuffix, this::withMultiplierPct);
            return this;
        }
    }
}
