package ru.yandex.direct.useractionlog.model;

import java.util.LinkedHashMap;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import ru.yandex.direct.binlogclickhouse.schema.FieldValue;
import ru.yandex.direct.binlogclickhouse.schema.FieldValueList;
import ru.yandex.direct.useractionlog.schema.ActionLogRecord;

import static ru.yandex.direct.dbschema.ppc.Ppc.PPC;

/**
 * Модель, представляющая стоку таблицы директа. Нужна, чтобы не
 * конвертировать FieldValueList в HashMap и обратно во время
 * применения цепочки стратегий.
 * <p>
 * Для проверки того, что модель не испортилась во время обработки,
 * каждая стратегия вызывает метод {@link RowModel#validate()}.
 * <p>
 * У каждой модели есть связанная с ней версия {@link RowModel#VERSION}.
 * Она нужна для записи в {@link ActionLogRecord} для того, чтобы Reader
 * мог работать с историческими данными.
 */
public abstract class RowModel {
    private static final String VERSION_FIELD = "%%version";
    private static final String VERSION = "0";
    protected LinkedHashMap<String, String> map;

    public RowModel() {
        this.map = new LinkedHashMap<>();
    }

    public RowModel(LinkedHashMap<String, String> map) {
        this.map = map;
    }

    public RowModel(FieldValueList list) {
        this.map = list.toMap();
    }

    public static String getVersionField() {
        return VERSION_FIELD;
    }

    public LinkedHashMap<String, String> getMap() {
        return map;
    }

    public String getVersion() {
        return VERSION;
    }

    public void setMap(LinkedHashMap<String, String> map) {
        this.map = map;
    }

    public FieldValueList toFieldValueList() {
        if (map.containsKey(getVersionField())) {
            if (!map.get(getVersionField()).equals(getVersion())) {
                throw new IllegalStateException(String.format(
                        "Already defined version %s in row model map and it is not equal to getVersion() = %s."
                                + " Map is %s", map.get(getVersionField()), getVersion(), map));
            }
            map.remove(getVersionField());
        }
        FieldValueList result = new FieldValueList(
                map.entrySet().stream().map(FieldValue::new).collect(Collectors.toList()));
        result.getFieldsValues().add(new FieldValue<>(getVersionField(), getVersion()));
        return result;
    }

    public void validate() {
    }

    public void clear() {
        map.clear();
    }

    public boolean isEmpty() {
        return map.isEmpty();
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null || getClass() != obj.getClass()) {
            return false;
        }
        RowModel that = (RowModel) obj;
        return Objects.equals(map, that.map);
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(map);
    }

    /**
     * Костыль, введённый ради DIRECT-75682.
     * <p>
     * Ранее предполагалось, что ObjectPath у объекта не может меняться, но оказалось, что это не всегда так.
     * В таком случае класс-потомок переопределяет этот метод.
     * <p>
     * При вызове этого метода считается, что старая модель - это this, а новая - это other.
     *
     * @param other Новая модель
     * @return true - если изменение в ObjectPath разрешено, false - если запрещено.
     */
    boolean isValidObjectPathChange(RowModel other) {
        return getObjectPathParts().equals(other.getObjectPathParts());
    }

    @Nonnull
    abstract Object getObjectPathParts();

    public static RowModel makeRowModel(String tableName, LinkedHashMap<String, String> values) {
        if (tableName.equals(PPC.BANNERS.getName())) {
            return new AdRowModel(values);
        } else if (tableName.equals(PPC.CAMP_OPTIONS.getName())) {
            return new CampOptionsRowModel(values);
        } else if (tableName.equals(PPC.CAMPAIGNS.getName())) {
            return new CampaignRowModel(values);
        } else if (tableName.equals(PPC.CLIENTS.getName())) {
            return new ClientRowModel(values);
        } else if (tableName.equals(PPC.CLIENTS_OPTIONS.getName())) {
            return new ClientOptionsRowModel(values);
        } else if (tableName.equals(PPC.PHRASES.getName())) {
            return new AdGroupRowModel(values);
        } else if (tableName.equals(PPC.RETARGETING_CONDITIONS.getName())) {
            return new RetargetingConditionsRowModel(values);
        } else if (tableName.equals(PPC.HIERARCHICAL_MULTIPLIERS.getName())) {
            return new HierarchicalMultiplierRowModel(values);
        } else if (tableName.equals(PPC.BIDS_BASE.getName())) {
            return new BidBaseRowModel(values);
        } else {
            throw new IllegalArgumentException("Unsupported table name " + tableName);
        }
    }

    public static RowModel makeRowModel(String tableName, FieldValueList values) {
        return makeRowModel(tableName, values.toMap());
    }

    @Override
    public String toString() {
        return "RowModel{" +
                "map=" + map +
                '}';
    }
}
