package ru.yandex.direct.useractionlog.schema;

import java.time.LocalDateTime;
import java.util.LinkedHashMap;
import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.binlog.reader.EnrichedDeleteRow;
import ru.yandex.direct.binlog.reader.EnrichedInsertRow;
import ru.yandex.direct.binlog.reader.EnrichedRow;
import ru.yandex.direct.binlog.reader.EnrichedUpdateRow;
import ru.yandex.direct.binlogclickhouse.schema.FieldValue;
import ru.yandex.direct.binlogclickhouse.schema.FieldValueList;
import ru.yandex.direct.tracing.data.DirectTraceInfo;
import ru.yandex.direct.useractionlog.model.RowModel;

@ParametersAreNonnullByDefault
public class ActionLogRecord {
    private final LocalDateTime dateTime;
    private final ObjectPath path;
    private final String gtid;
    private final int querySerial;
    private final int rowSerial;
    private final DirectTraceInfo directTraceInfo;
    private final String db;
    private final String type;
    private final Operation operation;
    private final FieldValueList oldFields;
    private final FieldValueList newFields;
    private final RecordSource recordSource;
    private final boolean deleted;

    public ActionLogRecord(LocalDateTime dateTime, ObjectPath path, String gtid, int querySerial, int rowSerial,
                           DirectTraceInfo directTraceInfo, String db, String type, Operation operation,
                           FieldValueList oldFields, FieldValueList newFields, RecordSource recordSource, boolean deleted) {
        this.dateTime = dateTime;
        this.path = path;
        this.gtid = gtid;
        this.querySerial = querySerial;
        this.rowSerial = rowSerial;
        this.directTraceInfo = directTraceInfo;
        this.db = db;
        this.type = type;
        this.operation = operation;
        this.oldFields = oldFields;
        this.newFields = newFields;
        this.recordSource = recordSource;
        this.deleted = deleted;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builderFrom(EnrichedRow row, RecordSource recordSource) {
        ActionLogRecord.Builder builder = new Builder()
                .withRecordSource(recordSource)
                .withDateTime(row.getDateTime())
                .withGtid(row.getGtid())
                .withQuerySerial(row.getQuerySerial())
                .withRowSerial(row.getRowSerial())
                .withDirectTraceInfo(row.getDirectTraceInfo())
                .withDb(row.getDbName())
                .withType(row.getTableName());
        if (row instanceof EnrichedInsertRow) {
            return builder
                    .withOperation(Operation.INSERT)
                    .withOldFields(FieldValueList.empty());
        } else if (row instanceof EnrichedUpdateRow) {
            return builder
                    .withOperation(Operation.UPDATE);
        } else if (row instanceof EnrichedDeleteRow) {
            return builder
                    .withOperation(Operation.DELETE)
                    .withNewFields(FieldValueList.empty());

        } else {
            throw new UnsupportedOperationException(row.getClass().toString());
        }
    }

    public LocalDateTime getDateTime() {
        return dateTime;
    }

    public ObjectPath getPath() {
        return path;
    }

    public String getGtid() {
        return gtid;
    }

    public int getQuerySerial() {
        return querySerial;
    }

    public int getRowSerial() {
        return rowSerial;
    }

    public DirectTraceInfo getDirectTraceInfo() {
        return directTraceInfo;
    }

    public String getDb() {
        return db;
    }

    public String getType() {
        return type;
    }

    public Operation getOperation() {
        return operation;
    }

    public FieldValueList getOldFields() {
        return oldFields;
    }

    public FieldValueList getNewFields() {
        return newFields;
    }

    public RowModel getOldModel() {
        return RowModel.makeRowModel(type, oldFields);
    }

    public RowModel getNewModel() {
        LinkedHashMap<String, String> map = newFields.toMap();
        for (FieldValue<String> fieldValue : oldFields.getFieldsValues()) {
            map.putIfAbsent(fieldValue.getName(), fieldValue.getValue());
        }
        return RowModel.makeRowModel(type, map);
    }

    public RecordSource getRecordSource() {
        return recordSource;
    }

    public boolean isDeleted() {
        return deleted;
    }

    public static class Builder {
        @Nullable
        private LocalDateTime dateTime;
        @Nullable
        private ObjectPath path;
        @Nullable
        private String gtid;
        @Nullable
        private Integer querySerial;
        @Nullable
        private Integer rowSerial;
        @Nullable
        private DirectTraceInfo directTraceInfo;
        @Nullable
        private String db;
        @Nullable
        private String type;
        @Nullable
        private Operation operation;
        @Nullable
        private FieldValueList oldFields;
        @Nullable
        private FieldValueList newFields;
        @Nullable
        private RecordSource recordSource;

        // Все поля обязательны к заполнению, но ради deleted сделано исключение, по умолчанию это false.
        // Сложно придумать ситуацию, когда writer должен будет создать запись с deleted=true.
        private boolean deleted = false;

        private Builder() {
        }

        @Nullable
        public Operation getOperation() {
            return operation;
        }

        @Nullable
        public FieldValueList getOldFields() {
            return oldFields;
        }

        @Nullable
        public FieldValueList getNewFields() {
            return newFields;
        }

        public Builder withDateTime(LocalDateTime dateTime) {
            this.dateTime = dateTime;
            return this;
        }

        public Builder withPath(ObjectPath path) {
            this.path = path;
            return this;
        }

        public Builder withGtid(String gtid) {
            this.gtid = gtid;
            return this;
        }

        public Builder withQuerySerial(int querySerial) {
            this.querySerial = querySerial;
            return this;
        }

        public Builder withRowSerial(int rowSerial) {
            this.rowSerial = rowSerial;
            return this;
        }

        public Builder withDirectTraceInfo(DirectTraceInfo directTraceInfo) {
            this.directTraceInfo = directTraceInfo;
            return this;
        }

        public Builder withDb(String db) {
            this.db = db;
            return this;
        }

        public Builder withType(String type) {
            this.type = type;
            return this;
        }

        public Builder withOperation(Operation operation) {
            this.operation = operation;
            return this;
        }

        public Builder withOldFields(FieldValueList oldFields) {
            this.oldFields = oldFields;
            return this;
        }

        public Builder withNewFields(FieldValueList newFields) {
            this.newFields = newFields;
            return this;
        }

        public Builder withRecordSource(RecordSource recordSource) {
            this.recordSource = recordSource;
            return this;
        }

        public Builder withDeleted(boolean deleted) {
            this.deleted = deleted;
            return this;
        }

        public ActionLogRecord build() {
            return new ActionLogRecord(
                    Objects.requireNonNull(dateTime, "Forgotten dateTime"),
                    Objects.requireNonNull(path, "Forgotten path"),
                    Objects.requireNonNull(gtid, "Forgotten gtid"),
                    Objects.requireNonNull(querySerial, "Forgotten querySerial"),
                    Objects.requireNonNull(rowSerial, "Forgotten rowSerial"),
                    Objects.requireNonNull(directTraceInfo, "Forgotten directTraceInfo"),
                    Objects.requireNonNull(db, "Forgotten db"),
                    Objects.requireNonNull(type, "Forgotten type"),
                    Objects.requireNonNull(operation, "Forgotten operation"),
                    Objects.requireNonNull(oldFields, "Forgotten oldFields"),
                    Objects.requireNonNull(newFields, "Forgotten newFields"),
                    Objects.requireNonNull(recordSource, "Forgotten recordSource"),
                    deleted);
        }
    }
}
