package ru.yandex.direct.useractionlog.schema;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.fasterxml.jackson.core.type.TypeReference;

import ru.yandex.direct.clickhouse.NestedField;
import ru.yandex.direct.clickhouse.SimpleField;
import ru.yandex.direct.clickhouse.TableSchema;
import ru.yandex.direct.clickhouse.types.BooleanClickHouseType;
import ru.yandex.direct.clickhouse.types.DateClickHouseType;
import ru.yandex.direct.clickhouse.types.DateTimeClickHouseType;
import ru.yandex.direct.clickhouse.types.EnumClickHouseType;
import ru.yandex.direct.clickhouse.types.IntBooleanClickHouseType;
import ru.yandex.direct.clickhouse.types.IntegerClickHouseType;
import ru.yandex.direct.clickhouse.types.LongClickHouseType;
import ru.yandex.direct.clickhouse.types.OptionalLongClickHouseType;
import ru.yandex.direct.clickhouse.types.OptionalStringClickHouseType;
import ru.yandex.direct.clickhouse.types.StringClickHouseType;
import ru.yandex.direct.clickhouse.types.UInt32ClickHouseType;

public class ActionLogSchema extends TableSchema {
    public static final SimpleField<LocalDate> DATE = new SimpleField<>("date", new DateClickHouseType());
    public static final SimpleField<LocalDateTime> DATETIME =
            new SimpleField<>("datetime", new DateTimeClickHouseType());
    public static final SimpleField<ObjectPath> PATH = new SimpleField<>("path", new ObjectPathClickHouseType());
    public static final SimpleField<String> GTID = new SimpleField<>("gtid", new StringClickHouseType());
    public static final SimpleField<Integer> QUERY_SERIAL =
            new SimpleField<>("query_serial", new UInt32ClickHouseType());
    public static final SimpleField<Integer> ROW_SERIAL = new SimpleField<>("row_serial", new UInt32ClickHouseType());
    public static final SimpleField<OptionalLong> REQID =
            new SimpleField<>("reqid", new OptionalLongClickHouseType("Int64", 0));
    public static final SimpleField<Optional<String>> METHOD =
            new SimpleField<>("method", new OptionalStringClickHouseType(""));
    public static final SimpleField<Optional<String>> SERVICE =
            new SimpleField<>("service", new OptionalStringClickHouseType(""));
    public static final SimpleField<OptionalLong> OPERATOR_UID =
            new SimpleField<>("operator_uid", new OptionalLongClickHouseType("Int64", 0));
    public static final SimpleField<String> DB = new SimpleField<>("db", new StringClickHouseType());

    /**
     * Тип записи. Часто совпадает с названием таблицы, но может содержать любую другую строку.
     * По типу таблицы должно быть возможно однозначно восстановить изменения из old/new_fields.
     */
    public static final SimpleField<String> TYPE = new SimpleField<>("type", new StringClickHouseType());
    public static final SimpleField<Operation> OPERATION =
            new SimpleField<>("operation", new EnumClickHouseType<>("Enum8", Operation.class));

    public static final NestedField OLD_FIELDS = new NestedField("old_fields");
    public static final NestedField.SubFieldArray<String> OLD_FIELDS_NAMES =
            OLD_FIELDS.add("name", new StringClickHouseType(), new TypeReference<List<String>>() {
            });
    public static final NestedField.SubFieldArray<String> OLD_FIELDS_VALUES =
            OLD_FIELDS.add("value", new StringClickHouseType(), new TypeReference<List<String>>() {
            });
    public static final NestedField.SubFieldArray<Boolean> OLD_FIELDS_NULLITIES =
            OLD_FIELDS.add("is_null", new BooleanClickHouseType(), new TypeReference<List<Boolean>>() {
            });

    public static final NestedField NEW_FIELDS = new NestedField("new_fields");
    public static final NestedField.SubFieldArray<String> NEW_FIELDS_NAMES =
            NEW_FIELDS.add("name", new StringClickHouseType(), new TypeReference<List<String>>() {
            });
    public static final NestedField.SubFieldArray<String> NEW_FIELDS_VALUES =
            NEW_FIELDS.add("value", new StringClickHouseType(), new TypeReference<List<String>>() {
            });
    public static final NestedField.SubFieldArray<Boolean> NEW_FIELDS_NULLITIES =
            NEW_FIELDS.add("is_null", new BooleanClickHouseType(), new TypeReference<List<Boolean>>() {
            });

    /**
     * Поддерживаются следующие значения:
     * <ul>
     * <li>{@link RecordSource#RECORD_SOURCE_DAEMON}
     * <li>{@link RecordSource#RECORD_SOURCE_MANUAL}
     * </ul>
     */
    public static final SimpleField<Integer> RECORD_SOURCE_TYPE =
            new SimpleField<>("record_source_type", new IntegerClickHouseType("Int8"));
    /**
     * Некая временная метка, позволяющая отличить старые данные от новых.
     * Для {@link RecordSource#RECORD_SOURCE_DAEMON} это дата-время старта приложения.
     * Для {@link RecordSource#RECORD_SOURCE_MANUAL} это дата-время начала вставки данных.
     * Как и в других полях, всё пишется в таймзоне UTC.
     */
    public static final SimpleField<LocalDateTime> RECORD_SOURCE_TIMESTAMP =
            new SimpleField<>("record_source_timestamp", new DateTimeClickHouseType());
    public static final SimpleField<Boolean> DELETED = new SimpleField<>("deleted", new IntBooleanClickHouseType());

    public static final SimpleField<Long> RECORD_VERSION = new SimpleField<>("record_version", new LongClickHouseType(
            String.format("Int64 MATERIALIZED %d * %s + toUnixTimestamp(%s)",
                    1L << 60, RECORD_SOURCE_TYPE.getExpr(), RECORD_SOURCE_TIMESTAMP.getExpr())));

    public static final SimpleField[] INDEX_COLUMNS = new SimpleField[]{
            DATE, PATH, GTID, QUERY_SERIAL, ROW_SERIAL,
    };

    public ActionLogSchema(String dbName, String tableName) {
        super(
                dbName,
                tableName,
                Arrays.asList(
                        DATE,
                        DATETIME,
                        PATH,
                        GTID,
                        QUERY_SERIAL,
                        ROW_SERIAL,
                        REQID,
                        METHOD,
                        SERVICE,
                        OPERATOR_UID,
                        DB,
                        TYPE,
                        OPERATION,
                        OLD_FIELDS,
                        NEW_FIELDS,
                        RECORD_SOURCE_TYPE,
                        RECORD_SOURCE_TIMESTAMP,
                        DELETED,
                        RECORD_VERSION
                ),
                "ReplacingMergeTree",
                Arrays.asList(
                        DATE.getName(),
                        "(" + Stream.of(INDEX_COLUMNS)
                                .map(SimpleField::getName)
                                .collect(Collectors.joining(", "))
                                + ")",
                        "1024",
                        RECORD_VERSION.getExpr()
                )
        );
    }
}
