package ru.yandex.direct.useractionlog.schema;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import ru.yandex.direct.useractionlog.AbstractId;
import ru.yandex.direct.useractionlog.AdGroupId;
import ru.yandex.direct.useractionlog.CampaignId;
import ru.yandex.direct.useractionlog.ClientId;

public abstract class ObjectPath {
    public static ObjectPath fromPathString(String pathString) {
        List<PathElement> elements = PathElement.parsePathString(pathString);
        if (elements.isEmpty()) {
            throw new IllegalArgumentException("Empty path string");
        }
        switch (elements.get(elements.size() - 1).getType()) {
            case CAMPAIGN:
                return CampaignPath.fromPathElements(elements);
            case ADGROUP:
                return AdGroupPath.fromPathElements(elements);
            case CLIENT:
                return ClientPath.fromPathElements(elements);
            case AD:
                return AdPath.fromPathElements(elements);
            default:
                throw new IllegalStateException(
                        "Path element of unexpected type " + elements.get(elements.size() - 1).getType() +
                                " in: " + elements
                );
        }
    }

    public abstract AbstractId getId();

    public abstract PathElementType getPathElementType();

    public abstract List<PathElement> toPathElements();

    public String toPathString() {
        return PathElement.toPathString(toPathElements());
    }

    @Override
    public boolean equals(Object other) {
        return other != null &&
                getClass() == other.getClass() &&
                Objects.equals(getId(), getClass().cast(other).getId());
    }

    @Override
    public int hashCode() {
        return Objects.hash(getId());
    }

    @Override
    public String toString() {
        return "ObjectPath." + this.getClass().getSimpleName() + "<" + toPathString() + ">";
    }

    public static class CampaignPath extends ChildPath<ClientPath, CampaignId> {
        public CampaignPath(ClientId clientId, CampaignId campaignId) {
            this(new ClientPath(clientId), campaignId);
        }

        public CampaignPath(ClientPath clientPath, CampaignId campaignId) {
            super(clientPath, campaignId, PathElementType.CAMPAIGN);
        }

        public ClientId getClientId() {
            return (ClientId) getParent().getId();
        }

        public ClientPath getClientPath() {
            return getParent();
        }

        static CampaignPath fromPathElements(List<PathElement> elements) {
            PathElement last = removeLast(elements);
            return new CampaignPath(ClientPath.fromPathElements(elements), new CampaignId(last.getId()));
        }
    }

    public static class ChildPath<P extends ObjectPath, I extends AbstractId> extends ObjectPath {
        private final P parent;
        private final I id;
        private final PathElementType type;

        ChildPath(P parent, I id, PathElementType type) {
            this.parent = parent;
            this.id = id;
            this.type = type;
        }

        protected P getParent() {
            return parent;
        }

        @Override
        public I getId() {
            return id;
        }

        @Override
        public PathElementType getPathElementType() {
            return type;
        }

        @Override
        public List<PathElement> toPathElements() {
            return inplaceConcat(parent.toPathElements(), new PathElement(getPathElementType(), getId()));
        }

        @Override
        public boolean equals(Object other) {
            return other != null &&
                    getClass() == other.getClass() &&
                    Objects.equals(getId(), getClass().cast(other).getId()) &&
                    Objects.equals(getParent(), getClass().cast(other).getParent());
        }

        @Override
        public int hashCode() {
            return Objects.hash(parent, getId());
        }
    }

    public static class AdGroupPath extends ChildPath<CampaignPath, AdGroupId> {
        public AdGroupPath(ClientId clientId, CampaignId campaignId, AdGroupId id) {
            this(new CampaignPath(clientId, campaignId), id);
        }

        public AdGroupPath(CampaignPath campaignPath, AdGroupId id) {
            super(campaignPath, id, PathElementType.ADGROUP);
        }

        static AdGroupPath fromPathElements(List<PathElement> elements) {
            PathElement last = removeLast(elements);
            return new AdGroupPath(CampaignPath.fromPathElements(elements), new AdGroupId(last.getId()));
        }

        public CampaignId getCampaignId() {
            return (CampaignId) this.getParent().getId();
        }

        public ObjectPath getCampaignPath() {
            return getParent();
        }

        public ClientId getClientId() {
            return getParent().getClientId();
        }
    }

    public static class AdPath extends ChildPath<AdGroupPath, AdId> {
        public AdPath(ClientId clientId, CampaignId campaignId, AdGroupId adGroupId, AdId id) {
            this(new AdGroupPath(clientId, campaignId, adGroupId), id);
        }

        public AdPath(AdGroupPath adGroupPath, AdId id) {
            super(adGroupPath, id, PathElementType.AD);
        }

        static AdPath fromPathElements(List<PathElement> elements) {
            PathElement last = ObjectPath.removeLast(elements);
            return new AdPath(AdGroupPath.fromPathElements(elements), new AdId(last.getId()));
        }

        public AdGroupId getAdGroupId() {
            return this.getParent().getId();
        }

        public CampaignId getCampaignId() {
            return this.getParent().getCampaignId();
        }

        public ClientId getClientId() {
            return this.getParent().getClientId();
        }

        public ObjectPath getAdGroupPath() {
            return getParent();
        }
    }

    public static class ClientPath extends ObjectPath {
        private final ClientId id;

        public ClientPath(ClientId id) {
            this.id = id;
        }

        @Override
        public AbstractId getId() {
            return id;
        }

        @Override
        public PathElementType getPathElementType() {
            return PathElementType.CLIENT;
        }

        @Override
        public List<PathElement> toPathElements() {
            return inplaceConcat(new ArrayList<>(), new PathElement(getPathElementType(), getId()));
        }

        static ClientPath fromPathElements(List<PathElement> elements) {
            if (elements.size() != 1) {
                throw new IllegalArgumentException("Client must be first element in path: " + elements);
            }
            return new ClientPath(new ClientId(elements.get(0).getId()));
        }
    }

    public static class RetargetingConditionPath extends ChildPath<ClientPath, RetargetingConditionId> {
        public RetargetingConditionPath(ClientPath clientPath, RetargetingConditionId id) {
            super(clientPath, id, PathElementType.RETARGETING_CONDITION);
        }

        public RetargetingConditionPath(ClientId clientId, RetargetingConditionId id) {
            this(new ClientPath(clientId), id);
        }

        public ClientId getClientId() {
            return (ClientId) getParent().getId();
        }

        public ClientPath getClientPath() {
            return getParent();
        }

        static CampaignPath fromPathElements(List<PathElement> elements) {
            PathElement last = removeLast(elements);
            return new CampaignPath(ClientPath.fromPathElements(elements), new CampaignId(last.getId()));
        }
    }

    public enum PathElementType {
        CAMPAIGN("camp"),
        ADGROUP("adgroup"),
        AD("ad"),
        CLIENT("client"),
        RETARGETING_CONDITION("ret_cond");

        private String prefix;

        private static final HashMap<String, PathElementType> byPrefix = new HashMap<>();

        static {
            for (PathElementType e : values()) {
                byPrefix.put(e.getPrefix(), e);
            }
        }

        PathElementType(String prefix) {
            this.prefix = prefix;
        }

        public String getPrefix() {
            return prefix;
        }

        public static PathElementType valueByPrefix(String prefix) {
            return byPrefix.get(prefix);
        }
    }

    public static class PathElement {
        private final PathElementType type;
        private final long id;

        PathElement(PathElementType type, long id) {
            this.type = type;
            this.id = id;
        }

        PathElement(PathElementType type, AbstractId id) {
            this(type, id.toLong());
        }

        static List<PathElement> parsePathString(String pathString) {
            return Arrays.stream(pathString.split("-"))
                    .filter(elementString -> !elementString.isEmpty())
                    .map(elementString -> {
                        String[] elementParts = elementString.split(":", 2);
                        return new PathElement(
                                PathElementType.valueByPrefix(elementParts[0]),
                                Long.parseLong(elementParts[1])
                        );
                    }).collect(Collectors.toList());
        }

        static String toPathString(List<PathElement> elements) {
            // Лишний "-" на конце конце позволяет искать в базе с помощью "path LIKE 'camp:123-%'".
            return String.join("-", elements.stream()
                    .map(PathElement::toPathElementString)
                    .collect(Collectors.toList())) + "-";
        }

        public PathElementType getType() {
            return type;
        }

        public long getId() {
            return id;
        }

        String toPathElementString() {
            return getType().getPrefix() + ":" + getId();
        }

        @Override
        public String toString() {
            return "PathElement{" +
                    "type=" + type +
                    ", id=" + id +
                    '}';
        }
    }

    static <T> List<T> inplaceConcat(List<T> list, T tail) {
        list.add(tail);
        return list;
    }

    static <T> T removeLast(List<T> list) {
        return list.remove(list.size() - 1);
    }
}
