package ru.yandex.direct.useractionlog.schema;

import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.Comparator;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * Структура, указывающая на то, каким образом строчка попала в таблицу пользовательских логов.
 * Позволяет приоритезировать записи с одинаковым первичным ключом.
 */
@ParametersAreNonnullByDefault
public class RecordSource implements Comparable<RecordSource> {
    /**
     * Источник изменения - приложение writer.
     */
    public static final Integer RECORD_SOURCE_DAEMON = 1;

    /**
     * Источник изменения - записи, внесённые вручную. Имеет более высокий приоритет, чем
     * {@link #RECORD_SOURCE_DAEMON}, чтобы можно было исправить то, что было сгенерировано приложением и чтобы
     * приложение точно никогда не перезаписало эти правки.
     */
    public static final Integer RECORD_SOURCE_MANUAL = 2;

    private final int type;
    private final LocalDateTime timestamp;

    public RecordSource(int type, LocalDateTime timestamp) {
        this.type = type;
        this.timestamp = timestamp;
    }

    /**
     * На отдельный запущенный процесс записи создаётся ровно один RecordSource и пишется во все записи.
     *
     * @return Экземпляр RecordSource для записи.
     */
    public static RecordSource makeDaemonRecordSource() {
        return new RecordSource(RECORD_SOURCE_DAEMON, LocalDateTime.now(ZoneId.of("UTC")));
    }

    public int getType() {
        return type;
    }

    public LocalDateTime getTimestamp() {
        return timestamp;
    }

    @Override
    public String toString() {
        return "RecordSource{" +
                "type=" + type +
                ", timestamp=" + timestamp +
                '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        RecordSource that = (RecordSource) o;
        return type == that.type &&
                Objects.equals(timestamp, that.timestamp);
    }

    @Override
    public int hashCode() {
        return Objects.hash(type, timestamp);
    }

    @Override
    public int compareTo(RecordSource o) {
        return Comparator
                .comparingInt(RecordSource::getType)
                .thenComparing(RecordSource::getTimestamp)
                .compare(this, o);
    }
}
