package ru.yandex.direct.useractionlog.schema;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;

import ru.yandex.direct.useractionlog.ChangeSource;
import ru.yandex.direct.useractionlog.Methods;
import ru.yandex.direct.useractionlog.Services;

/**
 * Класс для дополнительной информации о {@link ActionLogRecord}, которую нельзя
 * получить из таблицы пользологов.
 */
@ParametersAreNonnullByDefault
public class RecordStats {
    // id из trunk/protected/API/Settings.pm
    private static final Map<String, ChangeSource> APP_ID_TO_CHANGE_SOURCE = ImmutableMap.of(
            "c906d379328e4541a6d06e4894f2a03b", ChangeSource.COMMANDER,
            "c135345132e9449dab6416f3cc34ffab", ChangeSource.COMMANDER,
            "389dfb36ef1f4a16a2cdb47bc4510c7f", ChangeSource.DIRECT_MOBILE_APP,
            "74a699cdfe004d6ca576cfe4a8bfd79a", ChangeSource.DIRECT_MOBILE_APP);

    private final ChangeSource changeSource;
    private final Boolean isChangedByRecommendation;

    public RecordStats(ChangeSource changeSource, Boolean isChangedByRecommendation) {
        this.changeSource = changeSource;
        this.isChangedByRecommendation = isChangedByRecommendation;
    }

    public ChangeSource getChangeSource() {
        return changeSource;
    }

    public Boolean getIsChangedByRecommendation() {
        return isChangedByRecommendation;
    }

    @Override
    public String toString() {
        return "RecordStats{" +
                "changeSource=" + changeSource +
                ", getIsChangedByRecommendation=" + isChangedByRecommendation +
                '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        RecordStats that = (RecordStats) o;
        return changeSource == that.changeSource &&
                Objects.equals(isChangedByRecommendation, that.isChangedByRecommendation);
    }

    @Override
    public int hashCode() {
        return Objects.hash(changeSource, isChangedByRecommendation);
    }

    public static RecordStats from(ActionLogRecord record, @Nullable String applicationId) {
        ChangeSource changeSource = ChangeSource.OTHER;

        Optional<String> service = record.getDirectTraceInfo().getService();
        Optional<String> method = record.getDirectTraceInfo().getMethod();
        OptionalLong operator = record.getDirectTraceInfo().getOperatorUid();

        if (service.isPresent()) {
            if (service.get().equals(Services.WEB) || service.get().equals(Services.PERL_WEB)) {
                if (method.isPresent() && Methods.XLS_METHODS.contains(method.get())) {
                    changeSource = ChangeSource.XLS;
                } else {
                    changeSource = ChangeSource.WEB;
                }
            } else if (Services.API_SERVICES.contains(service.get())) {
                if (applicationId != null && APP_ID_TO_CHANGE_SOURCE.containsKey(applicationId)) {
                    changeSource = APP_ID_TO_CHANGE_SOURCE.get(applicationId);
                } else {
                    changeSource = ChangeSource.API_APP;
                }
            } else if (service.get().equals(Services.INTAPI) &&
                    method.isPresent() && Methods.WEB_INTAPI_METHODS.contains(method.get())) {
                changeSource = ChangeSource.WEB;
            } else if (service.get().equals(Services.INTAPI) &&
                    operator.orElse(0) > 0) {
                // Если вызывался intapi с operator_uid - значит это был вызов из perl-web
                changeSource = ChangeSource.WEB;
            } else if (service.get().equals(Services.SCRIPT) &&
                    method.isPresent() && method.get().equals(Methods.PPC_CAMP_QUEUE)) {
                changeSource = ChangeSource.WEB;
            } else if (service.get().equals(Services.JOBS) &&
                    method.isPresent() && method.get().equals(Methods.RECOMMENDATIONS_AUTO_APPLY_JOB)) {
                changeSource = ChangeSource.AUTO_RECOMMENDATION;
            }
        }

        Boolean isChangedByRecommendation = method.isPresent() && Methods.RECOMMENDATION_METHODS.contains(method.get());

        return new RecordStats(changeSource, isChangedByRecommendation);
    }
}
