package ru.yandex.direct.useractionlog.schema.dict;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.useractionlog.dict.DictDataCategory;

/**
 * Класс для объекта-записи в таблице {@link DictSchema}
 */
@ParametersAreNonnullByDefault
public class DictRecord {
    @Nonnull
    private final DictDataCategory category;

    private final String shard;

    private final long id;

    @Nonnull
    private final String value;

    @Nonnull
    private final LocalDateTime lastUpdated;

    public DictRecord(DictDataCategory category, String shard, long id, String value,
                      @Nullable LocalDateTime lastUpdated) {
        this.category = category;
        this.shard = shard;
        this.id = id;
        this.value = value;
        this.lastUpdated = lastUpdated != null ? lastUpdated : LocalDateTime.of(1970, 1, 1, 0, 0);
    }

    /**
     * Категория словарных данных
     */
    @Nonnull
    public DictDataCategory getCategory() {
        return category;
    }

    /**
     * Название шарда, например {@code ppc:10}
     */
    public String getShard() {
        return shard;
    }

    /**
     * Идентификатор, связанный с категорией
     */
    public long getId() {
        return id;
    }

    /**
     * Сами словарные данные в текстовом представлении
     */
    @Nonnull
    public String getValue() {
        return value;
    }

    /**
     * Дата и время обновления записи
     */
    @Nonnull
    public LocalDateTime getLastUpdated() {
        return lastUpdated;
    }

    /**
     * Дата обновления записи, для партиционирования таблицы
     */
    @Nonnull
    public LocalDate getDate() {
        // Поле не участвует ни в каких запросах и нужно только для партиционирования. Чем меньше партиций в таблице,
        // тем быстрее поиск по индексу в таблице. Если брать дату из lastUpdated, то партиций точно будет больше хотя
        // бы на этапе инициализации словаря.
        return ZonedDateTime.now(ZoneOffset.UTC).toLocalDate();
    }

    @Override
    public String toString() {
        return "DictRecord{" +
                "category=" + category +
                ", shard='" + shard + '\'' +
                ", id=" + id +
                ", value='" + value + '\'' +
                ", lastUpdated=" + lastUpdated +
                '}';
    }
}
