package ru.yandex.direct.useractionlog.schema.dict;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.EnumSet;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableBiMap;
import com.google.common.collect.ImmutableList;

import ru.yandex.direct.clickhouse.ClickHouseField;
import ru.yandex.direct.clickhouse.SimpleField;
import ru.yandex.direct.clickhouse.TableSchema;
import ru.yandex.direct.clickhouse.types.DateClickHouseType;
import ru.yandex.direct.clickhouse.types.DateTimeClickHouseType;
import ru.yandex.direct.clickhouse.types.LongClickHouseType;
import ru.yandex.direct.clickhouse.types.MappingClickHouseType;
import ru.yandex.direct.clickhouse.types.StringClickHouseType;
import ru.yandex.direct.useractionlog.dict.DictDataCategory;

/**
 * ClickHouse-таблица для словарных данных. Используется тот факт, что в Директовых таблицах у любого объекта есть
 * уникальный идентификатор независимо от его иерархии.
 * <br/>
 * Ключом таблицы является пара тип+id, значением - что угодно.
 */
@ParametersAreNonnullByDefault
public class DictSchema extends TableSchema {
    public static final SimpleField<String> SHARD = new SimpleField<>("shard", new StringClickHouseType());
    public static final SimpleField<Long> ID = new SimpleField<>("id", new LongClickHouseType("Int64"));
    public static final SimpleField<String> VALUE = new SimpleField<>("value", new StringClickHouseType());
    /**
     * Дата и время в UTC
     */
    public static final SimpleField<LocalDateTime> LAST_UPDATED =
            new SimpleField<>("last_updated", new DateTimeClickHouseType());
    /**
     * Дата в UTC
     */
    public static final SimpleField<LocalDate> DATE = new SimpleField<>("date", new DateClickHouseType());
    /**
     * Соответствие {@link DictDataCategory} и некоего уникального числа, которое записывается в кликхаус.
     * Число не указывается в самом {@link DictDataCategory}, потому что это детали реализации конкретного хранилища.
     * Не используются enum'ы от кликхауса, т.к. миграция схемы в кликхаусе при добавлении новой категории
     * словарных данных будет трудна.
     */
    private static final ImmutableBiMap<DictDataCategory, Integer> CATEGORY_INDEX_MAP =
            ImmutableBiMap.<DictDataCategory, Integer>builder()
                    .put(DictDataCategory.CAMPAIGN_PATH, 1)
                    .put(DictDataCategory.CAMPAIGN_NAME, 2)
                    .put(DictDataCategory.ADGROUP_NAME, 3)
                    // Идентификатор 4 использовался для хранения состояния временного словаря. Больше не нужен.
                    .put(DictDataCategory.ADGROUP_PATH, 5)
                    .put(DictDataCategory.AD_PATH, 6)
                    .put(DictDataCategory.AD_TITLE, 7)
                    .put(DictDataCategory.HIERARCHICAL_MULTIPLIERS_RECORD, 8)
                    .put(DictDataCategory.RETARGETING_CONDITION_NAME, 9)
                    .put(DictDataCategory.CAMPAIGN_DISABLED_IPS, 10)
                    .put(DictDataCategory.CAMPAIGN_DISABLED_SSP, 11)
                    .put(DictDataCategory.CAMPAIGN_DONT_SHOW, 12)
                    .put(DictDataCategory.CAMPAIGN_GEO, 13)
                    .put(DictDataCategory.CAMPAIGN_MINUS_WORDS, 14)
                    .put(DictDataCategory.CAMPAIGN_TIME_TARGET, 15)
                    .put(DictDataCategory.CAMPAIGN_STRATEGY_DATA, 16)
                    .put(DictDataCategory.ADGROUP_GEO, 17)
                    .build();
    public static final SimpleField<DictDataCategory> TYPE =
            new SimpleField<>("type", new MappingClickHouseType<>("Int8", CATEGORY_INDEX_MAP));

    static {
        Preconditions.checkState(Objects.equals(
                CATEGORY_INDEX_MAP.keySet(),
                EnumSet.allOf(DictDataCategory.class)));
    }

    public DictSchema(String dbName, String tableName) {
        super(dbName,
                tableName,
                new ImmutableList.Builder<ClickHouseField>()
                        .add(TYPE, SHARD, ID, VALUE, LAST_UPDATED, DATE)
                        .build(),
                "ReplacingMergeTree",
                new ImmutableList.Builder<String>()
                        .add(DATE.getExpr())
                        .add(String.format("(%s)", Stream.of(TYPE, SHARD, ID)
                                .map(SimpleField::getExpr)
                                .collect(Collectors.joining(", "))))
                        .add("8192")
                        .add(LAST_UPDATED.getExpr())
                        .build());
    }
}
