package ru.yandex.direct.useractionlog.writer.generator;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.useractionlog.dict.DictDataCategory;
import ru.yandex.direct.useractionlog.schema.RecordSource;

import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;

/*

=== Суточная статистика по изменениям в таблице баннеров ===

Статистика взята из продакшновой базы, которую пишет приложение binlog-to-clickhouse-writer.

Запрос:

SELECT
    row.name,
    operation,
    count() / 7,
    sum(length(row.value)) / 7
FROM binlog_rows
ARRAY JOIN
    row.name,
    row.value
WHERE (date >= '2017-08-14') AND (date <= '2017-08-20') AND (db = 'ppc') AND (table = 'banners')
GROUP BY
    row.name,
    operation
ORDER BY
    row.name ASC,
    operation ASC
FORMAT CSV


Ответ:

"BannerID","INSERT",2664905.5714285714,2664905.5714285714
"BannerID","UPDATE",1989208,19892080
"LastChange","INSERT",2664905.5714285714,55963017
"LastChange","UPDATE",100923401.14285715,2119391424
"banner_type","INSERT",2664905.5714285714,11043794.142857144
"body","INSERT",2664905.5714285714,322857147.71428573
"body","UPDATE",2748440.5714285714,316744937
"cid","INSERT",2664905.5714285714,21310575.42857143
"domain","INSERT",2664905.5714285714,41762818.428571425
"domain","UPDATE",393562,6732157.571428572
"domain_id","INSERT",2664905.5714285714,0
"flags","INSERT",2650728.4285714286,2828633.5714285714
"flags","UPDATE",11097876,476842696.28571427
"geoflag","INSERT",2664905.5714285714,2664459.8571428573
"geoflag","UPDATE",198062.85714285713,198062.85714285713
"href","INSERT",2664905.5714285714,573914265.4285715
"href","UPDATE",4078895.4285714286,1450492166.857143
"opts","INSERT",2664905.5714285714,11760212
"opts","UPDATE",202962.2857142857,971852
"phoneflag","INSERT",2664905.5714285714,8440552.57142857
"phoneflag","UPDATE",18684951.42857143,88778975.42857143
"pid","INSERT",2664905.5714285714,26628630.85714286
"pid","UPDATE",1740.857142857143,17408.571428571428
"reverse_domain","INSERT",2664905.5714285714,41762818.428571425
"reverse_domain","UPDATE",393562,6732157.571428572
"sitelinks_set_id","INSERT",2664905.5714285714,19626847.85714286
"sitelinks_set_id","UPDATE",3943599.5714285714,35406879.428571425
"statusActive","INSERT",2664905.5714285714,5329811.142857143
"statusActive","UPDATE",6431502.857142857,16131412.142857144
"statusArch","INSERT",2664905.5714285714,5366219.428571428
"statusArch","UPDATE",1062947.142857143,2919354.1428571427
"statusBsSynced","INSERT",2664905.5714285714,5329811.142857143
"statusBsSynced","UPDATE",128786471.42857143,554592873.7142857
"statusMetricaStop","INSERT",2664905.5714285714,5329811.142857143
"statusModerate","INSERT",2664905.5714285714,8636491.857142856
"statusModerate","UPDATE",34831298.428571425,165806160.7142857
"statusPostModerate","INSERT",2664905.5714285714,5652481.428571428
"statusPostModerate","UPDATE",14427595.285714285,38452053.85714286
"statusShow","INSERT",2664905.5714285714,7948058.285714285
"statusShow","UPDATE",1949810.7142857143,4610507.428571428
"statusSitelinksModerate","INSERT",2664905.5714285714,8502607.714285715
"statusSitelinksModerate","UPDATE",34353528.28571428,163696428
"title","INSERT",2664905.5714285714,125319208.85714285
"title","UPDATE",1412500.4285714286,59016196.571428575
"title_extension","INSERT",2060046.142857143,17118693
"title_extension","UPDATE",1031385,43352500
"type","INSERT",2664905.5714285714,18429862.85714286
"vcard_id","INSERT",2664905.5714285714,12310126.857142856
"vcard_id","UPDATE",1549209.857142857,10639441.857142856
"yacontextCategories","INSERT",2664905.5714285714,52665.857142857145
"yacontextCategories","UPDATE",1177.142857142857,0

График:

from matplotlib import pyplot
import pandas as pd
df = pd.read_csv('file.csv', names=['Field', 'Operation', 'Count', 'Bytes'])
fig, axes = pyplot.subplots(nrows=1, ncols=2)
for i, (base, col) in enumerate([(10, 'Count'), (2, 'Bytes')]):
    df.pivot(index='Field', columns='Operation', values=col) \
      .loc[sorted(set(df.Field), reverse=True, key=str.lower)] \
      .plot.barh(stacked=True, ax=axes[i])
    axes[i].set_title('Avg Daily ' + col)
    axes[i].set_xscale('log', basex=base)
    if i != 0:
        axes[i].set_ylabel("")
pyplot.show()

По графику видны самые часто изменяемые поля (в порядке убывания):
* statusBsSynced (сто миллионов)
* LastChange (сто миллионов)
* statusModerate (десятки миллионов)
* statusSitelinksModerate (десятки миллионов)
* phoneflag (десять миллионов)
* statusPostModerate (десять миллионов)
* flags (десять миллионов)

При этом занимаемое место на диске тратят другие поля (в порядке убывания):
* LastChange (пара гигабайт)
* href (пара гигабайт)
* body (пол гигабайта)
* statusBsSynced (пол гигабайта)
* flags (пол гигабайта, однако оно состоит из константных строк, разделённых запятой - должно хорошо сжиматься)
* title (сто мегабайт)
* status*Moderate (десятки мегабайт)

 */
@ParametersAreNonnullByDefault
class AdRowProcessingStrategy extends ModularRowProcessingStrategy {
    AdRowProcessingStrategy(RecordSource recordSource) {
        super(recordSource, new AdPathStrategy(), new FieldsStrategyChain(
                new FilterFieldsStrategy(
                        // время последней модификации баннера, завязано на экспорт данных в фокусировщик и в методе
                        // API::GetChanges (обычно время не обновляется при обновлении невидимых параметров баннера)
                        "LastChange",

                        // статус, описывающий синхронность данных уровня кампании в Директе и БК
                        "statusBsSynced",

                        // результат модерации текста объявления
                        "statusModerate",

                        // statusPostModerate
                        "statusPostModerate",

                        // Результат модерации быстрых ссылок объявлений
                        "statusSitelinksModerate",

                        // результат модерации визитки
                        "phoneflag",

                        // устанавливается в экспорте в БК на основе ответа; означает может ли баннер показываться
                        // при выполнении других условий показа
                        "statusActive",

                        // уникальный идентификатор набора
                        "sitelinks_set_id",

                        // id визитки
                        "vcard_id",

                        // id баннера в Баннерной Крутилке
                        "BannerID"),
                StringsFromDictStrategy.builder()
                        .withIdField(BANNERS.BID.getName())
                        .with(DictDataCategory.AD_TITLE, BANNERS.TITLE.getName())
                        .build(),
                DeduplicationFieldsStrategy.builder().build()));
    }
}
