package ru.yandex.direct.useractionlog.writer.generator;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiConsumer;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.binlog.reader.EnrichedRow;
import ru.yandex.direct.useractionlog.dict.DictRepository;
import ru.yandex.direct.useractionlog.dict.DictRequest;
import ru.yandex.direct.useractionlog.dict.DictRequestsFiller;
import ru.yandex.direct.useractionlog.dict.DictResponsesAccessor;
import ru.yandex.direct.useractionlog.dict.FreshDictValuesFiller;

/**
 * Работа со словарями при пакетной обработке {@link EnrichedRow}.
 */
@ParametersAreNonnullByDefault
public class BatchRowDictProcessing {
    private BatchRowDictProcessing() {
    }

    /**
     * Получить необходимые словарные данные и записать новые словарные данные для обработки пачки {@link EnrichedRow}.
     *
     * @param dictRepository Репозиторий словарных данных. <b>Внимание:</b> Подразумевается, что используемый
     *                       dictRepository умеет группировать данные в пачки и записывать одним разом. Процедура много
     *                       раз вызывает {@link DictRepository#addData(Map)} с небольшими данными.
     * @param dictFiller     Некий объект, который запрашивает словарные данные, которые потом понадобятся для
     *                       обработки кортежей и извлекает из кортежей новые словарные данные.
     * @param rows           Сами кортежи.
     * @param errorWrapper   Обёртка, которая ловит ошибки и добавляет дополнительную информацию, упрощающую поиск ошибки.
     */
    public static Result handleEvents(DictRepository dictRepository,
                                      DictFiller dictFiller,
                                      List<EnrichedRow> rows,
                                      BiConsumer<EnrichedRow, Runnable> errorWrapper) {
        // Собираются все запросы на получение словарных данных и одной пачкой забираются из словаря.
        //
        // Затем в цикле для каждого кортежа:
        // * Извлечь словарные данные из кортежа и добавить в словарь.
        // * Положить кортеж либо в список обработанных, либо в список необработанных.
        //
        // Каждая итерация цикла должна иметь доступ к словарным данным, которые были получены на предыдущих итерациях.
        //
        // Когда обработанные кортежи отделяются от необработанных, для обработанных кортежей создаются
        // DictResponsesAccessor, для необработанных - списки необработанных запросов.
        Set<DictRequest> dictRequests = new HashSet<>();
        List<Map<DictRequest, ?>> dictResponsesForRow = new ArrayList<>(rows.size());
        for (EnrichedRow row : rows) {
            // Хитрость: на этом этапе нужно создать Set, в который положат какие-то объекты (запросы в словарь).
            // Далее потребуется создать Map с точно такими же ключами (запрос-ответ).
            // Так что ниже создаётся Map, он превращается в Set благодаря Collections.newSetFromMap,
            // а затем при заполнении ответов в исходном Map меняются значения на нужные. Экономия на клонировании
            // хешмапов в количестве rows.size(). Нюанс в том, что Collections.newSetFromMap требует Map<K, Boolean>,
            // а далее нужен Map<K, Object>. Потому далее обманываем компилятор с SuppressWarnings, зная, что
            // преобразование Map<K, Boolean> в Map<K, Object> безопасно.
            Map<DictRequest, Boolean> currentRowDictResponses = new HashMap<>();
            dictResponsesForRow.add(currentRowDictResponses);
            DictRequestsFiller dictRequestsFiller =
                    new DictRequestsFiller(dictRequests, Collections.newSetFromMap(currentRowDictResponses));
            errorWrapper.accept(row, () -> dictFiller.fillDictRequests(row, dictRequestsFiller));
        }
        Preconditions.checkState(rows.size() == dictResponsesForRow.size(), "Bug. Different lists sizes.");

        Map<DictRequest, Object> fetchedData = dictRepository.getData(Collections.unmodifiableSet(dictRequests));
        Map<DictRequest, Object> newData = new HashMap<>();

        Result result = new Result();
        Iterator<Map<DictRequest, ?>> currentRowDictResponsesIterator = dictResponsesForRow.iterator();
        for (EnrichedRow row : rows) {
            @SuppressWarnings("unchecked")
            Map<DictRequest, Object> actualDataForRow = (Map<DictRequest, Object>) currentRowDictResponsesIterator.next();
            // В продакшне не должно быть никаких необработанных запросов. Их наличие свидетельствует о багах.
            // Будем считать, что багов нет (надеюсь, это было смешно), и создаём массив с capacity=0.
            List<DictRequest> unprocessedDictRequests = new ArrayList<>(0);
            for (DictRequest dictRequest : actualDataForRow.keySet()) {
                Object value = fetchedData.get(dictRequest);
                if (value == null) {
                    unprocessedDictRequests.add(dictRequest);
                } else {
                    actualDataForRow.put(dictRequest, value);
                }
            }
            // Далее обязательно следует копировать все данные, а не складывать в результирующий список
            // ленивые обёртки типа Maps.filterKeys и Sets.filter. Наружу возвращается множество dictResponseAccessor с
            // этими данными. Каждый из этих dictResponseAccessor должен содержать в себе данные, актуальные для
            // сопоставленного с ним кортежа. Если не копировать, то в середине списка будет доступ к данным из будущего.
            if (unprocessedDictRequests.isEmpty()) {
                DictResponsesAccessor dictResponsesAccessor = new DictResponsesAccessor(actualDataForRow);
                errorWrapper.accept(row, () -> dictFiller.fillFreshDictValues(
                        row, dictResponsesAccessor, new FreshDictValuesFiller(newData)));
                if (!newData.isEmpty()) {
                    // Каждая итерация цикла должна иметь доступ к словарным данным, которые были получены на предыдущих
                    // итерациях. Поэтому новые данные дополняют или перезаписывают данные, полученные из словаря.
                    fetchedData.putAll(newData);
                    dictRepository.addData(newData);
                    newData.clear();
                }
                result.processed.add(Pair.of(row, dictResponsesAccessor));
            } else {
                result.unprocessed.add(Pair.of(row, unprocessedDictRequests));
            }
        }
        return result;
    }

    public static class Result {
        /**
         * Список пар для кортежей с необработанными словарными данными (т.е. {@link DictFiller} запросил такие
         * словарные данные, которых нет в базе.
         * <p>
         * Первый элемент пары - сам кортеж, второй элемент - список запросов, на которые не были получены ответы.
         */
        public final List<Pair<EnrichedRow, Collection<DictRequest>>> unprocessed = new ArrayList<>();

        /**
         * Список пар для кортежей с обработанными словарными данными.
         * <p>
         * Первый элемента пары - сам кортеж, второй элемент - объект для доступа к словарным данным, которые запросил
         * для него {@link DictFiller}.
         */
        public final List<Pair<EnrichedRow, DictResponsesAccessor>> processed = new ArrayList<>();
    }

}
