package ru.yandex.direct.useractionlog.writer.generator;

import java.util.Arrays;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.useractionlog.dict.DictDataCategory;
import ru.yandex.direct.useractionlog.schema.RecordSource;

import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;

/*
Повторяемость результата запроса не гарантируется, но выборка достаточно большая, чтобы составить по ней
статистику.

SELECT
    arrayJoin(row.name) AS changed_field,
    count() AS count
FROM binlog_rows
PREWHERE (table = 'campaigns') AND (operation = 'UPDATE') AND (date >= today())
GROUP BY changed_field
ORDER BY count DESC

┌─changed_field────────────┬────count─┐
│ sum_spent_units          │ 40705714 │
│ shows                    │ 40704552 │
│ lastShowTime             │ 40350685 │
│ sum_spent                │  4376192 │
│ clicks                   │  4293881 │
│ statusBsSynced           │  3063910 │
│ balance_tid              │   827668 │
│ sum                      │   821804 │
│ sum_units                │   789750 │
│ LastChange               │   165692 │
│ timeTarget               │   104084 │
│ disabled_ssp             │   102039 │
│ DontShow                 │   102039 │
│ name                     │   101247 │
│ disabledIps              │   101247 │
│ statusActive             │    79996 │
│ statusYacobotDeleted     │    65941 │
│ sum_last                 │    51435 │
│ statusShow               │    42235 │
│ statusMail               │    26508 │
│ sum_to_pay               │    25043 │
│ archived                 │    21059 │
│ statusModerate           │    21014 │
│ statusBsArchived         │    17793 │
│ autobudgetForecastDate   │     9490 │
│ opts                     │     8918 │
│ statusEmpty              │     8431 │
│ day_budget               │     8198 │
│ timezone_id              │     7768 │
│ rf                       │     7725 │
│ wallet_cid               │     7449 │
│ OrderID                  │     7255 │
│ strategy_min_price       │     6595 │
│ autoOptimization         │     6308 │
│ ContextLimit             │     5699 │
│ autobudgetForecast       │     5224 │
│ dontShowYacontext        │     3655 │
│ AgencyUID                │     3488 │
│ geo                      │     2879 │
│ autobudget_sum           │     1715 │
│ strategy_no_premium      │     1426 │
│ autobudget               │     1274 │
│ day_budget_show_mode     │     1264 │
│ autobudget_avg_bid       │     1132 │
│ statusAutobudgetForecast │      936 │
│ statusOpenStat           │      923 │
│ ContextPriceCoef         │      919 │
│ autobudget_bid           │      851 │
│ start_time               │      509 │
│ finish_time              │      482 │
│ statusBehavior           │      371 │
│ AgencyID                 │      356 │
│ ManagerUID               │      260 │
│ autobudget_limit_clicks  │      154 │
│ uid                      │      122 │
│ autobudget_avg_cpa       │       75 │
│ autobudget_goal_id       │       63 │
│ rfReset                  │       13 │
│ currencyConverted        │       12 │
│ autobudget_date          │       10 │
│ paid_by_certificate      │        5 │
└──────────────────────────┴──────────┘

По поводу изменения полей timeTarget и timezone_id:

SELECT
    count(),
    sum(if(has(row.name, 'timeTarget') AND NOT has(row.name, 'timezone_id'), 1, 0)) AS onlyTimeTarget,
    sum(if(NOT has(row.name, 'timeTarget') AND has(row.name, 'timezone_id'), 1, 0)) AS onlyTimezoneId,
    sum(if(has(row.name, 'timeTarget') AND has(row.name, 'timezone_id'), 1, 0)) AS both
FROM binlog_rows
WHERE (db = 'ppc') AND (table = 'campaigns') AND (operation = 'UPDATE') AND (date >= '2017-11-26') AND (date < '2017-12-26')

┌────count()─┬─onlyTimeTarget─┬─onlyTimezoneId─┬───both─┐
│ 2707170335 │        9962659 │              0 │ 472175 │
└────────────┴────────────────┴────────────────┴────────┘

Поле timeTarget текстовое и не приезжает в noblob в before, поле timezone_id - smallint(5) unsigned и оно приезжает в
noblob. Поля timezone_id и timeTarget связаны по смыслу, не должно быть записей, в которых есть данные только об одном
из этих полей. Поэтому если timezone_id поменялся, а timeTarget не поменялся, нам необходимо загрузить timeTarget из
словаря.
*/
@ParametersAreNonnullByDefault
class CampaignRowProcessingStrategy extends ModularRowProcessingStrategy {
    CampaignRowProcessingStrategy(RecordSource recordSource) {
        // В таблице кампаний происходит много изменений полей, которые точно не интересны в пользовательских логах.
        // Например, много запросов вида `update campaigns set shows = ... where cid = ...`. Информация об этих полях
        // выбрасывается до записи в clickhouse.
        super(recordSource, new CampaignPathStrategy(), new FieldsStrategyChain(
                new FilterFieldsStrategy(
                        // Время последнего изменения кампании
                        "LastChange",

                        // суммарное количество засчитанных кликов
                        "clicks",

                        // приблизительное время последнего показа рекламной кампании
                        "lastShowTime",

                        // суммарное количество засчитанных показов
                        "shows",

                        // статус, описывающий синхронность данных уровня кампании в Директе и БК
                        "statusBsSynced",

                        // Указывает на то, проводится ли модерация или с каким результатом завершилась
                        "statusModerate",

                        // должна ли быть кампания удалена из базы фокусировщика
                        "statusYacobotDeleted",

                        // потрачено на кампании, в валюте кампании
                        "sum_spent",

                        // потрачено на кампании, в юнитах (для Директа - смысла не имеет, для Баяна - показы)),
                        "sum_spent_units"),
                StringsFromDictStrategy.builder()
                        .withIdField(CAMPAIGNS.CID.getName())
                        .with(DictDataCategory.CAMPAIGN_NAME, CAMPAIGNS.NAME.getName())
                        .with(DictDataCategory.CAMPAIGN_TIME_TARGET, CAMPAIGNS.TIME_TARGET.getName())
                        .writeInUpdateOnly(DictDataCategory.CAMPAIGN_TIME_TARGET)
                        .with(DictDataCategory.CAMPAIGN_STRATEGY_DATA, CAMPAIGNS.STRATEGY_DATA.getName())
                        .with(DictDataCategory.CAMPAIGN_GEO, CAMPAIGNS.GEO.getName())
                        .withFieldGroup(CAMPAIGNS.TIME_TARGET.getName(),
                                CAMPAIGNS.TIMEZONE_ID.getName())
                        .build(),
                CollectionsDiffStrategy.builder()
                        .withIdField(CAMPAIGNS.CID.getName())
                        .withCommaDelimitedListCategory(DictDataCategory.CAMPAIGN_DISABLED_IPS,
                                CAMPAIGNS.DISABLED_IPS.getName())
                        .withJsonListCategory(DictDataCategory.CAMPAIGN_DISABLED_SSP,
                                CAMPAIGNS.DISABLED_SSP.getName())
                        .withCommaDelimitedListCategory(DictDataCategory.CAMPAIGN_DONT_SHOW,
                                CAMPAIGNS.DONT_SHOW.getName())
                        // Для поля geo важен порядок следования элементов. CollectionsDiffStrategy
                        // в данный момент не может обеспечить восстановление исходного порядка следования элементов.
                        .build(),
                DeduplicationFieldsStrategy.builder()
                        .withFieldGroup(CAMPAIGNS.TIME_TARGET.getName(),
                                CAMPAIGNS.TIMEZONE_ID.getName())
                        // Так вышло, что настройки сетей и стратегии используют одни и те же поля.
                        // Например, оба смотрят в platform.
                        .withFieldGroup(Arrays.asList(
                                Pair.of(CAMPAIGNS.ATTRIBUTION_MODEL.getName(), value -> value),
                                Pair.of(CAMPAIGNS.AUTOBUDGET.getName(), value -> value),
                                Pair.of(CAMPAIGNS.CONTEXT_LIMIT.getName(), value -> value),
                                Pair.of(CAMPAIGNS.CONTEXT_PRICE_COEF.getName(), value -> value),
                                Pair.of(CAMPAIGNS.CURRENCY.getName(), value -> value),
                                Pair.of(CAMPAIGNS.DAY_BUDGET.getName(), value -> value),
                                Pair.of(CAMPAIGNS.DAY_BUDGET_SHOW_MODE.getName(), value -> value),
                                Pair.of(CAMPAIGNS.OPTS.getName(),
                                        (String value) -> Arrays.asList(value.split(","))
                                                .contains("enable_cpc_hold")),
                                Pair.of(CAMPAIGNS.PLATFORM.getName(), value -> value),
                                Pair.of(CAMPAIGNS.STRATEGY_DATA.getName(), value -> value),
                                Pair.of(CAMPAIGNS.STRATEGY_NAME.getName(), value -> value),
                                Pair.of(CAMPAIGNS.TYPE.getName(), value -> value)))
                        // По поводу особенностей временного таргетинга смотри комментарий выше
                        .withFieldGroup(
                                CAMPAIGNS.RF.getName(),
                                CAMPAIGNS.RF_RESET.getName())
                        .build()));
    }
}
