package ru.yandex.direct.useractionlog.writer.generator;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.direct.binlog.reader.EnrichedRow;
import ru.yandex.direct.binlog.reader.EnrichedUpdateRow;
import ru.yandex.direct.binlog.reader.MySQLSimpleRowIndexed;
import ru.yandex.direct.useractionlog.dict.DictDataCategory;
import ru.yandex.direct.useractionlog.dict.DictRequestsFiller;
import ru.yandex.direct.useractionlog.dict.DictResponsesAccessor;
import ru.yandex.direct.useractionlog.dict.FreshDictValuesFiller;
import ru.yandex.direct.useractionlog.model.RowModelPair;
import ru.yandex.direct.utils.JsonUtils;

/**
 * Стратегия, когда есть поля, хранящие в себе коллекции каких-то объектов, и следует показать изменения в этой
 * коллекции. Модифицирует только события UPDATE.
 * <p>
 * Если поле изменилось, то в before пишется старое значение поля, новое значение удаляется из after, а вместо него
 * добавляются поля {@code <name>__before} и {@code <name>__after}, которые содержат удалённые и добавленные
 * элементы коллекции соответственно.
 * <p>
 * Пример:
 * <p>
 * Пусть в кампании был список минус-фраз: {@code minus_words = ["гвозди", "груши", "яблоки"]}.
 * <p>
 * Новое значение списка минус-фраз: {@code minus_words = ["гвозди", "болты"]}.
 * <p>
 * Старого значения не будет в событии об изменении, поэтому оно будет взято из словаря и добавлено в before.
 * <p>
 * Новое значение будет удалено из after, а вместо него будут добавлены две новые колонки:
 * {@code minus_words__removed = ["груши", "яблоки"]},
 * {@code minus_words__added = ["болты"]}.
 * <p>
 * Если после получения словарного значения выяснится, что фактически коллекция не изменилась, то она будет удалена из
 * after и не будет добавлена в before, т.е. результат подобен применению {@link DeduplicationFieldsStrategy} к
 * noblob-полям.
 */
@ParametersAreNonnullByDefault
public class CollectionsDiffStrategy implements FieldsStrategy, DictFiller {
    private final Map<String, Info> infoByFieldName;
    private final String idField;

    private CollectionsDiffStrategy(Map<String, Info> infoByFieldName, String idField) {
        this.infoByFieldName = Collections.unmodifiableMap(infoByFieldName);
        this.idField = idField;
    }

    static Builder builder() {
        return new CollectionsDiffStrategy.Builder();
    }

    @Override
    public void fillDictRequests(EnrichedRow row, DictRequestsFiller dictRequests) {
        if (row instanceof EnrichedUpdateRow) {
            MySQLSimpleRowIndexed before = ((EnrichedUpdateRow) row).getFields().getBefore();
            MySQLSimpleRowIndexed after = ((EnrichedUpdateRow) row).getFields().getAfter();
            long id = Util.dataForGettingId(row).getByName(idField).getValueAsLong();
            infoByFieldName.forEach((key, value) -> {
                if (!before.hasName(key) && after.hasName(key)) {
                    dictRequests.require(value.category, id);
                }
            });
        }
    }

    @Override
    public void handleUpdate(RowModelPair pair, DictResponsesAccessor dictResponsesAccessor) {
        LinkedHashMap<String, String> beforeMap = new LinkedHashMap<>(pair.before.getMap());
        LinkedHashMap<String, String> afterMap = new LinkedHashMap<>(pair.after.getMap());
        Set<String> addedFromDict = new HashSet<>();
        long id = Long.parseLong(beforeMap.get(idField));

        for (Map.Entry<String, Info> infoEntry : infoByFieldName.entrySet()) {
            String name = infoEntry.getKey();
            if (!beforeMap.containsKey(name) && afterMap.containsKey(name)) {
                beforeMap.put(name, (String) dictResponsesAccessor.get(infoEntry.getValue().category, id));
                addedFromDict.add(name);
            }
        }

        LinkedHashMap<String, String> newAfter = new LinkedHashMap<>();
        for (Map.Entry<String, String> afterEntry : afterMap.entrySet()) {
            String name = afterEntry.getKey();
            Info info = infoByFieldName.get(name);
            if (info != null) {
                LinkedHashSet<String> setFromBefore = info.deserializer.apply(beforeMap.get(name));
                LinkedHashSet<String> setFromAfter = info.deserializer.apply(afterEntry.getValue());
                if (!setFromBefore.equals(setFromAfter)) {
                    newAfter.put(
                            name + "__removed",
                            info.serializer.apply(setFromBefore.stream()
                                    .filter(s -> !setFromAfter.contains(s))
                                    .collect(Collectors.toList())));
                    newAfter.put(
                            name + "__added",
                            info.serializer.apply(setFromAfter.stream()
                                    .filter(s -> !setFromBefore.contains(s))
                                    .collect(Collectors.toList())));
                } else if (addedFromDict.remove(name)) {
                    beforeMap.remove(name);
                }
            } else {
                newAfter.put(name, afterEntry.getValue());
            }
        }

        if (!addedFromDict.isEmpty()) {
            pair.before.setMap(beforeMap);
        }
        pair.after.setMap(newAfter);
        pair.validate();
    }

    @Override
    public void fillFreshDictValues(EnrichedRow row, DictResponsesAccessor dictData,
                                    FreshDictValuesFiller freshDictValues) {
        MySQLSimpleRowIndexed rowData = Util.dataForFillingDict(row);
        if (rowData != null) {
            Map<String, String> data = rowData.toMap();
            long id = Long.parseLong(data.get(idField));
            data.forEach((name, value) -> {
                Info info = infoByFieldName.get(name);
                if (info != null) {
                    if (value == null) {
                        value = info.serializer.apply(Collections.emptyList());
                    }
                    freshDictValues.add(info.category, id, value);
                }
            });
        }
    }

    @Override
    public DictFiller makePureDictFiller() {
        return new DictFiller() {
            @Override
            public void fillDictRequests(EnrichedRow row, DictRequestsFiller dictRequests) {
                // По умолчанию этот класс запрашивает словарные данные, чтобы вставить название в событие
                // изменения/удаления объекта. Если стоит цель только сгенерировать новые словарные данные, то ничего
                // запрашивать не надо.
            }

            @Override
            public void fillFreshDictValues(EnrichedRow row, DictResponsesAccessor dictData,
                                            FreshDictValuesFiller freshDictValues) {
                CollectionsDiffStrategy.this.fillFreshDictValues(row, dictData, freshDictValues);
            }

            @Override
            public Collection<DictDataCategory> provides() {
                return CollectionsDiffStrategy.this.provides();
            }
        };
    }

    private static class Info {
        final DictDataCategory category;
        final Function<Collection<String>, String> serializer;
        final Function<String, LinkedHashSet<String>> deserializer;

        private Info(DictDataCategory category, Function<Collection<String>, String> serializer,
                     Function<String, LinkedHashSet<String>> deserializer) {
            this.category = category;
            this.serializer = serializer;
            this.deserializer = deserializer;
        }
    }

    static class Builder {
        private final Map<String, Info> infoByFieldName;
        private String idField;

        private Builder() {
            // Повторяемость результатов - хорошо для тестирования. У простого HashMap может быть непредсказуемый
            // порядок ключей при итерации, а у LinkedHashMap он предсказуемый.
            infoByFieldName = new LinkedHashMap<>();
        }

        private static LinkedHashSet<String> deserializeCommaDelimitedList(String raw) {
            return new LinkedHashSet<>(StringUtils.isEmpty(raw)
                    ? Collections.emptyList()
                    : Arrays.asList(raw.split(",")));
        }

        private static String serializeCommaDelimitedList(Collection<String> collection) {
            return String.join(",", collection);
        }

        @SuppressWarnings("unchecked")
        private static LinkedHashSet<String> deserializeJsonList(String raw) {
            return new LinkedHashSet<>(StringUtils.isEmpty(raw)
                    ? Collections.emptyList()
                    : (List<String>) JsonUtils.fromJson(raw, List.class));
        }

        private void checkWasNotInserted(DictDataCategory categoryCandidate, String nameFieldCandidate) {
            infoByFieldName.forEach((name, info) -> {
                if (name.equals(nameFieldCandidate)) {
                    throw new IllegalArgumentException(
                            "Field name " + name + " already used for category " + info.category);
                }
                if (info.category == categoryCandidate) {
                    throw new IllegalArgumentException(
                            "Category " + categoryCandidate + " already used for field name " + name);
                }
            });
        }

        /**
         * @param idField Название колонки, в которой хранится идентификатор для словаря.
         */
        Builder withIdField(String idField) {
            this.idField = idField;
            return this;
        }

        /**
         * Следует работать с определённой категорией, которая хранится в колонке с определённым названием.
         * Формат данных - строки, разделённые запятой.
         */
        Builder withCommaDelimitedListCategory(DictDataCategory category, String nameField) {
            checkWasNotInserted(category, nameField);
            infoByFieldName.put(nameField, new Info(category,
                    Builder::serializeCommaDelimitedList,
                    Builder::deserializeCommaDelimitedList));
            return this;
        }

        /**
         * Следует работать с определённой категорией, которая хранится в колонке с определённым названием.
         * Формат данных - список строк в JSON.
         */
        Builder withJsonListCategory(DictDataCategory category, String nameField) {
            checkWasNotInserted(category, nameField);
            infoByFieldName.put(nameField, new Info(category,
                    JsonUtils::toJson,
                    Builder::deserializeJsonList));
            return this;
        }

        CollectionsDiffStrategy build() {
            return new CollectionsDiffStrategy(infoByFieldName, idField);
        }
    }

    @Override
    public Collection<DictDataCategory> provides() {
        return infoByFieldName.values().stream()
                .map(i -> i.category)
                .collect(Collectors.toList());
    }
}
