package ru.yandex.direct.useractionlog.writer.generator;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;

import ru.yandex.direct.binlog.reader.EnrichedRow;
import ru.yandex.direct.useractionlog.dict.DictDataCategory;
import ru.yandex.direct.useractionlog.dict.DictResponsesAccessor;
import ru.yandex.direct.useractionlog.dict.FreshDictValuesFiller;
import ru.yandex.direct.useractionlog.model.RowModelPair;

import static ru.yandex.direct.useractionlog.writer.generator.Util.getGroupsByFieldName;

/**
 * При изменении записи в таблице оставляет только те значения, которые изменились. При создании или удалении записи в
 * таблице ничего не меняет.
 */
@ParametersAreNonnullByDefault
class DeduplicationFieldsStrategy implements FieldsStrategy {

    private final boolean keepEverythingInBefore;
    private final ImmutableSet<String> alwaysWriteFieldInBefore;
    private final ImmutableMap<String, ImmutableList<Group>> groupsByFieldName;

    private DeduplicationFieldsStrategy(boolean keepEverythingInBefore,
                                        ImmutableSet<String> alwaysWriteFieldInBefore,
                                        Collection<Group> groups) {
        this.keepEverythingInBefore = keepEverythingInBefore;
        this.alwaysWriteFieldInBefore = alwaysWriteFieldInBefore;
        this.groupsByFieldName = getGroupsByFieldName(groups);
    }

    public static Builder builder() {
        return new Builder();
    }

    @Override
    public void handleUpdate(RowModelPair pair, DictResponsesAccessor dictResponsesAccessor) {
        Map<String, String> beforeMap = pair.before.getMap();
        Map<String, String> afterMap = pair.after.getMap();
        cleanUpBefore(beforeMap, afterMap);
        cleanUpAfter(beforeMap, afterMap);
        pair.validate();
    }

    private void cleanUpBefore(Map<String, String> beforeMap, Map<String, String> afterMap) {
        if (keepEverythingInBefore) {
            return;
        }
        Set<String> fieldsToObserve = new HashSet<>(beforeMap.keySet());
        while (!fieldsToObserve.isEmpty()) {
            String fieldName = fieldsToObserve.iterator().next();
            fieldsToObserve.remove(fieldName);
            boolean shouldKeepFieldInBefore = alwaysWriteFieldInBefore.contains(fieldName)
                    || !Objects.equals(beforeMap.get(fieldName), afterMap.get(fieldName));
            if (groupsByFieldName.containsKey(fieldName)) {
                Iterator<Group> groups = groupsByFieldName.get(fieldName).iterator();
                while (!shouldKeepFieldInBefore && groups.hasNext()) {
                    Group group = groups.next();
                    shouldKeepFieldInBefore = !group.extract(beforeMap).equals(group.extract(afterMap));
                    if (shouldKeepFieldInBefore) {
                        fieldsToObserve.removeAll(group.getNames());
                    }
                }
            }
            if (!shouldKeepFieldInBefore) {
                beforeMap.remove(fieldName);
                // Не может быть таких полей, чтобы они были в after и отсутстовали в before.
                // Если поле удаляется из before, то оно удаляется и из after.
                afterMap.remove(fieldName);
            }
        }
    }

    private void cleanUpAfter(Map<String, String> beforeMap, Map<String, String> afterMap) {
        // Если поле есть в after и отсутствует в before, то такое поле следует оставить.
        // Поэтому ниже специально итерируется по before, чтобы не рассматривать эти случаи.
        for (Map.Entry<String, String> fieldEntry : beforeMap.entrySet()) {
            String fieldName = fieldEntry.getKey();
            boolean shouldRemoveFieldFromAfter = afterMap.containsKey(fieldName)
                    && Objects.equals(fieldEntry.getValue(), afterMap.get(fieldName));
            if (shouldRemoveFieldFromAfter) {
                afterMap.remove(fieldName);
            }
        }
    }

    @Override
    public void fillFreshDictValues(EnrichedRow row, DictResponsesAccessor dictData,
                                    FreshDictValuesFiller freshDictValues) {
        // При удалении дублей никакие словарные данные не создаются
    }

    @Override
    public Collection<DictDataCategory> provides() {
        return Collections.emptyList();
    }

    @Override
    public DictFiller makePureDictFiller() {
        return DummyFiller.INSTANCE;
    }

    private static class GroupWithExtractors implements Group {
        /**
         * [(fieldName, fieldValue -> object with equals)]
         */
        private final ImmutableMap<String, Function<String, Object>> fieldHandlers;

        GroupWithExtractors(ImmutableMap<String, Function<String, Object>> fieldHandlers) {
            this.fieldHandlers = fieldHandlers;
        }

        @Override
        public List<Object> extract(Map<String, String> fieldMap) {
            List<Object> result = new ArrayList<>(fieldHandlers.size());
            for (Map.Entry<String, Function<String, Object>> fieldHandler : fieldHandlers.entrySet()) {
                if (fieldMap.containsKey(fieldHandler.getKey())) {
                    result.add(fieldHandler.getValue().apply(fieldMap.get(fieldHandler.getKey())));
                } else {
                    result.add(NOTHING);
                }
            }
            return result;
        }

        @Override
        public Collection<String> getNames() {
            return fieldHandlers.keySet();
        }
    }

    public static class Builder {
        private final ImmutableSet.Builder<String> alwaysWriteFieldInBeforeBuilder = ImmutableSet.builder();
        private final ImmutableList.Builder<Group> groupsBuilder = ImmutableList.builder();
        private boolean keepEverythingInBefore = false;

        /**
         * Не удалять никакие поля из before, только из after.
         */
        public Builder keepsEverythingInBefore() {
            keepEverythingInBefore = true;
            return this;
        }

        /**
         * @param fields Список полей, которые никогда не следует удалять из before.
         */
        public Builder alwaysWriteFieldsInBefore(String... fields) {
            alwaysWriteFieldInBeforeBuilder.addAll(Arrays.asList(fields));
            return this;
        }

        /**
         * Объявляет группу полей. Если хотя бы одно из полей в группе изменилось, то все эти поля останутся в before.
         * В after значения будут записываться как обычно - только изменившиеся, без учёта групп.
         */
        public Builder withFieldGroup(String... fields) {
            groupsBuilder.add(new GroupAsIs(ImmutableList.copyOf(fields)));
            return this;
        }

        /**
         * Объявляет группу полей. Если хотя бы одно из полей в группе изменилось, то все эти поля останутся в before.
         *
         * @param fieldExtractors Пара поле-функция. Функция преобразовывает строку из кортежа в некий объект. Если
         *                        результаты этой функции будут отличаться для значений определённого поля из before и
         *                        after, то в before будет оставлена вся группа. В after значения будут записываться как
         *                        обычно - только изменившиеся, без учёта групп.
         */
        public Builder withFieldGroup(List<Map.Entry<String, Function<String, Object>>> fieldExtractors) {
            groupsBuilder.add(new GroupWithExtractors(ImmutableMap.copyOf(fieldExtractors)));
            return this;
        }

        public DeduplicationFieldsStrategy build() {
            return new DeduplicationFieldsStrategy(
                    keepEverythingInBefore,
                    alwaysWriteFieldInBeforeBuilder.build(),
                    groupsBuilder.build());
        }
    }
}
