package ru.yandex.direct.useractionlog.writer.generator;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.binlog.reader.EnrichedRow;
import ru.yandex.direct.binlogclickhouse.schema.FieldValueList;
import ru.yandex.direct.useractionlog.dict.DictResponsesAccessor;
import ru.yandex.direct.useractionlog.model.RowModel;
import ru.yandex.direct.useractionlog.model.RowModelPair;
import ru.yandex.direct.useractionlog.schema.Operation;

/**
 * Генерация данных для записи в {@link ru.yandex.direct.useractionlog.schema.ActionLogSchema#OLD_FIELDS}
 * и {@link ru.yandex.direct.useractionlog.schema.ActionLogSchema#NEW_FIELDS}.
 * Разделено на стадии ради эффективной работы со словарными данными.
 */
@ParametersAreNonnullByDefault
interface FieldsStrategy extends DictFiller, PureDictFillerFactory {
    /**
     * Генерация данных для записи в {@link ru.yandex.direct.useractionlog.schema.ActionLogSchema#NEW_FIELDS}.
     * Специфично для события создания записи в таблице.
     * Гарантируется, что метод не будет вызван, если доступны не все словарные данные, запрошенные в
     * {@link #fillDictRequests(EnrichedRow, DictRequestsFiller)}.
     *
     * @param after                 Исходные данные
     * @param dictResponsesAccessor Словарные данные, запрошенные ранее.
     */
    default void handleInsert(RowModel after, DictResponsesAccessor dictResponsesAccessor) {
    }

    /**
     * Генерация данных для записи в {@link ru.yandex.direct.useractionlog.schema.ActionLogSchema#OLD_FIELDS}
     * и {@link ru.yandex.direct.useractionlog.schema.ActionLogSchema#NEW_FIELDS}.
     * Специфично для события изменения записи в таблице.
     * Гарантируется, что метод не будет вызван, если доступны не все словарные данные, запрошенные в
     * {@link #fillDictRequests(EnrichedRow, DictRequestsFiller)}.
     *
     * @param pair                  Исходные данные со значениями до и после изменения записи в таблице.
     * @param dictResponsesAccessor Словарные данные, запрошенные ранее.
     */
    default void handleUpdate(RowModelPair pair,
                              DictResponsesAccessor dictResponsesAccessor) {
    }

    /**
     * Генерация данных для записи в {@link ru.yandex.direct.useractionlog.schema.ActionLogSchema#OLD_FIELDS}.
     * Специфично для события удаления записи в таблице.
     * Гарантируется, что метод не будет вызван, если доступны не все словарные данные, запрошенные в
     * {@link #fillDictRequests(EnrichedRow, DictRequestsFiller)}.
     *
     * @param before                Исходные данные
     * @param dictResponsesAccessor Словарные данные, запрошенные ранее.
     */
    default void handleDelete(RowModel before, DictResponsesAccessor dictResponsesAccessor) {
    }

    /**
     * Выполняет нужную ветвь в зависимости от указанной операции. Возвращает пару кортежей, которые следует записать в
     * {@link ru.yandex.direct.useractionlog.schema.ActionLogRecord.Builder#withOldFields(FieldValueList)} и
     * {@link ru.yandex.direct.useractionlog.schema.ActionLogRecord.Builder#withNewFields(FieldValueList)}.
     *
     * @param operation             Какую ветвь выбрать для обработки. Не обязательно должна совпадать с исходным типом
     *                              кортежа из бинлога.
     * @param pair                  Входные данные. Для INSERT значение в before будет проигнорировано, для DELETE будет
     *                              проигнорирован after.
     * @param dictResponsesAccessor Словарные данные, запрошенные ранее.
     */
    default void handle(Operation operation, RowModelPair pair,
                        DictResponsesAccessor dictResponsesAccessor) {
        switch (operation) {
            case INSERT:
                pair.before.clear();
                handleInsert(pair.after, dictResponsesAccessor);
                return;
            case UPDATE:
                handleUpdate(pair, dictResponsesAccessor);
                return;
            case DELETE:
                pair.after.clear();
                handleDelete(pair.before, dictResponsesAccessor);
                return;
            default:
                throw new UnsupportedOperationException(operation.toString());
        }
    }
}
