package ru.yandex.direct.useractionlog.writer.generator;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;

import ru.yandex.direct.binlog.reader.EnrichedRow;
import ru.yandex.direct.useractionlog.dict.DictRequestsFiller;
import ru.yandex.direct.useractionlog.dict.DictResponsesAccessor;
import ru.yandex.direct.useractionlog.model.RowModel;
import ru.yandex.direct.useractionlog.model.RowModelPair;

/**
 * Генератор, комбинирующий несколько потомков {@link FieldsStrategy}.
 * Для всех методов {@link FieldsStrategy} будут вызываться аналогичные методы от указанных в конструкторе объектов
 * и в том же порядке, в котором они указаны в конструкторе.
 */
@ParametersAreNonnullByDefault
class FieldsStrategyChain extends DictFillerChain implements FieldsStrategy {
    private final FieldsStrategy[] handlers;

    FieldsStrategyChain(FieldsStrategy... handlers) {
        super(handlers);
        this.handlers = handlers;
    }

    @Override
    public void fillDictRequests(EnrichedRow row, DictRequestsFiller dictRequests) {
        for (FieldsStrategy handler : handlers) {
            handler.fillDictRequests(row, dictRequests);
        }
    }

    @Override
    public void handleInsert(RowModel after, DictResponsesAccessor dictResponsesAccessor) {
        for (FieldsStrategy handler : handlers) {
            handler.handleInsert(after, dictResponsesAccessor);
        }
    }

    @Override
    public void handleUpdate(RowModelPair pair, DictResponsesAccessor dictResponsesAccessor) {
        for (FieldsStrategy handler : handlers) {
            handler.handleUpdate(pair, dictResponsesAccessor);
        }
    }

    @Override
    public void handleDelete(RowModel before, DictResponsesAccessor dictResponsesAccessor) {
        for (FieldsStrategy handler : handlers) {
            handler.handleDelete(before, dictResponsesAccessor);
        }
    }

    @Override
    public DictFiller makePureDictFiller() {
        ImmutableList.Builder<DictFiller> builder = ImmutableList.builder();
        for (FieldsStrategy handler : handlers) {
            builder.add(handler.makePureDictFiller());
        }
        return new DictFillerChain(builder.build());
    }
}
