package ru.yandex.direct.useractionlog.writer.generator;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.lang3.tuple.Triple;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.binlog.reader.EnrichedDeleteRow;
import ru.yandex.direct.binlog.reader.EnrichedInsertRow;
import ru.yandex.direct.binlog.reader.EnrichedRow;
import ru.yandex.direct.binlog.reader.EnrichedUpdateRow;
import ru.yandex.direct.binlog.reader.MySQLSimpleRowIndexed;
import ru.yandex.direct.dbschema.ppc.enums.HierarchicalMultipliersType;
import ru.yandex.direct.useractionlog.AdGroupId;
import ru.yandex.direct.useractionlog.dict.DictDataCategory;
import ru.yandex.direct.useractionlog.dict.DictRequestsFiller;
import ru.yandex.direct.useractionlog.dict.DictResponsesAccessor;
import ru.yandex.direct.useractionlog.dict.FreshDictValuesFiller;
import ru.yandex.direct.useractionlog.model.HierarchicalMultiplierRowModel;
import ru.yandex.direct.useractionlog.model.HierarchicalMultipliersData;
import ru.yandex.direct.useractionlog.model.RowModelPair;
import ru.yandex.direct.useractionlog.schema.ActionLogRecord;
import ru.yandex.direct.useractionlog.schema.HierarchicalMultiplierId;
import ru.yandex.direct.useractionlog.schema.ObjectPath;
import ru.yandex.direct.useractionlog.schema.Operation;
import ru.yandex.direct.useractionlog.schema.RecordSource;

import static ru.yandex.direct.dbschema.ppc.Ppc.PPC;

/*

SELECT
    table,
    row.name AS row_name,
    operation,
    round(count() / 7) AS daily_count,
    formatReadableSize(sum(length(row.value)) / 7) AS daily_size_bytes
FROM binlog_rows
ARRAY JOIN
    row.name,
    row.value
WHERE (date >= '2017-08-14') AND (date <= '2017-08-20') AND (db = 'ppc') AND (table LIKE '%multiplier%')
GROUP BY
    table,
    row_name,
    operation
ORDER BY
    table ASC,
    row_name ASC,
    operation ASC

-- 29 rows in set. Elapsed: 123.523 sec. Processed 24.90 billion rows, 3.06 TB (201.60 million rows/s., 24.75 GB/s.)

┌─table─────────────────────────┬─row_name───────────────────┬─operation─┬─daily_count─┬─daily_size_bytes─┐
│ demography_multiplier_values  │ age                        │ INSERT    │       23337 │ 95.85 KiB        │
│ demography_multiplier_values  │ gender                     │ INSERT    │       23337 │ 69.92 KiB        │
│ demography_multiplier_values  │ hierarchical_multiplier_id │ INSERT    │       23337 │ 205.11 KiB       │
│ demography_multiplier_values  │ last_change                │ INSERT    │       23337 │ 638.12 KiB       │
│ demography_multiplier_values  │ last_change                │ UPDATE    │      430644 │ 11.50 MiB        │
│ demography_multiplier_values  │ multiplier_pct             │ INSERT    │       23337 │ 47.87 KiB        │
│ demography_multiplier_values  │ multiplier_pct             │ UPDATE    │      430645 │ 1.03 MiB         │
│ geo_multiplier_values         │ hierarchical_multiplier_id │ INSERT    │         935 │ 8.22 KiB         │
│ geo_multiplier_values         │ last_change                │ INSERT    │         935 │ 25.58 KiB        │
│ geo_multiplier_values         │ last_change                │ UPDATE    │         345 │ 9.45 KiB         │
│ geo_multiplier_values         │ multiplier_pct             │ INSERT    │         935 │ 2.48 KiB         │
│ geo_multiplier_values         │ multiplier_pct             │ UPDATE    │         345 │ 982.43 B         │
│ geo_multiplier_values         │ region_id                  │ INSERT    │         935 │ 3.89 KiB         │
│ hierarchical_multipliers      │ cid                        │ INSERT    │       28070 │ 219.07 KiB       │
│ hierarchical_multipliers      │ is_enabled                 │ INSERT    │       28070 │ 27.41 KiB        │
│ hierarchical_multipliers      │ is_enabled                 │ UPDATE    │         163 │ 163.29 B         │
│ hierarchical_multipliers      │ last_change                │ INSERT    │       28070 │ 767.53 KiB       │
│ hierarchical_multipliers      │ last_change                │ UPDATE    │      297828 │ 7.95 MiB         │
│ hierarchical_multipliers      │ multiplier_pct             │ INSERT    │       28070 │ 45.61 KiB        │
│ hierarchical_multipliers      │ multiplier_pct             │ UPDATE    │      296865 │ 859.16 KiB       │
│ hierarchical_multipliers      │ pid                        │ INSERT    │       28070 │ 170.03 KiB       │
│ hierarchical_multipliers      │ syntetic_key_hash          │ INSERT    │       28070 │ 531.69 KiB       │
│ hierarchical_multipliers      │ type                       │ INSERT    │       28070 │ 509.49 KiB       │
│ retargeting_multiplier_values │ hierarchical_multiplier_id │ INSERT    │        7962 │ 69.97 KiB        │
│ retargeting_multiplier_values │ last_change                │ INSERT    │        7962 │ 217.70 KiB       │
│ retargeting_multiplier_values │ last_change                │ UPDATE    │         645 │ 17.63 KiB        │
│ retargeting_multiplier_values │ multiplier_pct             │ INSERT    │        7962 │ 21.54 KiB        │
│ retargeting_multiplier_values │ multiplier_pct             │ UPDATE    │         645 │ 1.84 KiB         │
│ retargeting_multiplier_values │ ret_cond_id                │ INSERT    │        7962 │ 46.59 KiB        │
└───────────────────────────────┴────────────────────────────┴───────────┴─────────────┴──────────────────┘

Для каждой записи генерируется большое словарное значение и большая строчка в логе. Это компенсируется тем, что самих изменений в этих таблицах ничтожно мало по сравнению с группами и баннерами.

 */

/**
 * Один общий обработчик для всех таблиц hierarchical_multipliers и *_multiplier_values.
 * Каждая строчка на выходе содержит в себе почти всё, что выдал бы запрос
 * <pre>
 *     SELECT hm.*, dmv.*, gmv.*, rmv.*, rc.condition_name
 *     FROM hierarchical_multipliers AS hm
 *     LEFT JOIN demography_multiplier_values.* AS dmv USING (hierarchical_multiplier_ids)
 *     LEFT JOIN geo_multiplier_values.* AS gmv USING (hierarchical_multiplier_ids)
 *     LEFT JOIN retargeting_multiplier_values.* AS rmv USING (hierarchical_multiplier_ids)
 *     LEFT JOIN retargeting_conditions AS rc ON rmv.ret_cond_id = rc.ret_cond_id
 *     WHERE hierarchical_multiplier_id = ?
 * </pre>
 * Предполагается, что при создании группы корректировок ставок запись в hierarchical_multipliers создаётся первой, а
 * при удалении запись из hierarchical_multipliers удаляется последней. как если бы таблицы были связаны через FOREIGN
 * KEY. Иначе этот код сломается.
 * <p>
 * Строчки из таблицы hierarchical_multipliers записываются без изменений.
 * Значения из таблиц *_multiplier_values записываются без изменений, а к названиям колонок добавляется суффикс
 * {@literal "_<internalId>"}.
 * internalId - некое натуральное число.
 * databaseId - значение PRIMARY KEY одной из таблиц *_multiplier_values.
 * <p>
 * Также добавляется несуществующая колонка "related_ids", которая содержит в себе пары internalId-databaseId,
 * разделённые запятой.
 * <p>
 * Также добавляется несуществующая колонка version = "01".
 * <p>
 * <table>
 * <tr>
 * <th>Таблица</th>
 * <th>Изменение</th>
 * <th>Результат</th>
 * </tr>
 * <tr>
 * <td>hierarchical_multipliers</td>
 * <td>INSERT</td>
 * <td>В логе появится INSERT-запись только если нет связанных таблиц (т.е. это mobile_multiplier/video_multipier).</td>
 * </tr>
 * <tr>
 * <td>hierarchical_multipliers</td>
 * <td>UPDATE</td>
 * <td>В лог пишется как UPDATE.</td>
 * </tr>
 * <tr>
 * <td>hierarchical_multipliers</td>
 * <td>DELETE</td>
 * <td>Если нет связанных таблиц, то пишется в лог как DELETE. Иначе в лог ничего не запишется.</td>
 * </tr>
 * <tr>
 * <td>*_multiplier_values</td>
 * <td>INSERT</td>
 * <td>В лог пишется запись, будто у всей группы произошёл UPDATE. В этой записи появится новая корректировка.</td>
 * </tr>
 * <tr>
 * <td>*_multiplier_values</td>
 * <td>UPDATE</td>
 * <td>Пишется как UPDATE, в котором меняются только те поля, которые относятся к изменившейся корректировке.</td>
 * </tr>
 * <tr>
 * <td>*_multiplier_values</td>
 * <td>DELETE</td>
 * <td>Если нет связанных таблиц, то пишется в лог как DELETE. Если есть связанные записи и при удалении корректировке
 * в
 * группе больше не остаётся корректировок, то тоже пишется в лог как DELETE. Если это была не последняя корректировка,
 * то в лог пишется как UPDATE, и об удалении корректировки можно будет узнать по изменившемуся полю
 * "related_ids".</td>
 * </tr>
 * </table>
 * <p>
 * Ремарка для корректировок по условиям показов. Если изменилось название условия показов, то в корректировке будет
 * отображаться старое название до тех пор, пока не изменится сама корректировка.
 * <p>
 * Общий принцип работы:
 * <p>
 * При каждом событии в одной из таблиц из словаря берётся слепок всей группы из словаря. В этом слепке хранится
 * состояние всей группы до пришедшего события. К этому слепку применяются изменения из события. Новый слепок
 * записывается в словарь и в лог. Следующее событие в одной из таблиц будет применяться уже к слепку, который был
 * записан на этой итерации.
 */
@ParametersAreNonnullByDefault
class HierarchicalMultipliersStrategy implements RowProcessingStrategy, DictFiller, PureDictFillerFactory {
    private static final Logger logger = LoggerFactory.getLogger(HierarchicalMultipliersStrategy.class);
    private static final String ACTION_LOG_RECORD_TYPE = "hierarchical_multipliers";

    /**
     * Таблицы, которые известно как обрабатывать
     */
    private static final Set<String> SUPPORTED_TABLES = ImmutableSet.of(
            PPC.DEMOGRAPHY_MULTIPLIER_VALUES.getName(),
            PPC.GEO_MULTIPLIER_VALUES.getName(),
            PPC.HIERARCHICAL_MULTIPLIERS.getName(),
            PPC.RETARGETING_MULTIPLIER_VALUES.getName());

    /**
     * Типы, которые изветсно как обрабатывать
     */
    private static final Set<String> SUPPORTED_TYPES = ImmutableSet.of(
            HierarchicalMultipliersType.demography_multiplier.getLiteral(),
            HierarchicalMultipliersType.geo_multiplier.getLiteral(),
            HierarchicalMultipliersType.mobile_multiplier.getLiteral(),
            HierarchicalMultipliersType.performance_tgo_multiplier.getLiteral(),
            HierarchicalMultipliersType.retargeting_multiplier.getLiteral(),
            HierarchicalMultipliersType.video_multiplier.getLiteral());

    /* Преобразования над кортежами, которые будут сделаны перед записью в лог/словарь */
    private static final FieldsStrategy fieldStrategy = new FieldsStrategyChain(
            new FilterFieldsStrategy(PPC.HIERARCHICAL_MULTIPLIERS.SYNTETIC_KEY_HASH.getName()),
            DeduplicationFieldsStrategy.builder().keepsEverythingInBefore().build());
    private final RecordSource recordSource;

    HierarchicalMultipliersStrategy(RecordSource recordSource) {
        this.recordSource = recordSource;
    }

    private static boolean isRootTable(EnrichedRow row) {
        return row.getTableName().equals(PPC.HIERARCHICAL_MULTIPLIERS.getName());
    }

    private static boolean rowCanBeHandled(EnrichedRow row) {
        if (isRootTable(row)) {
            String type = Util.dataForGettingId(row)
                    .getByName(PPC.HIERARCHICAL_MULTIPLIERS.TYPE.getName())
                    .getValueAsString();
            return SUPPORTED_TYPES.contains(type);
        } else {
            return SUPPORTED_TABLES.contains(row.getTableName());
        }
    }

    private static boolean rootHasRelatedTable(MySQLSimpleRowIndexed rowData) {
        return !HierarchicalMultipliersData.TYPES_WITHOUT_RELATED_TABLES.contains(
                rowData.getByName(PPC.HIERARCHICAL_MULTIPLIERS.TYPE.getName()).getValueAsString());
    }

    private static HierarchicalMultiplierId getHierarchicalMultiplierId(MySQLSimpleRowIndexed rowData) {
        return new HierarchicalMultiplierId(
                Util.fieldAsLong(rowData, PPC.HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID));
    }

    private static long getRelatedId(EnrichedRow row, MySQLSimpleRowIndexed rowData) {
        long relatedId;
        if (row.getTableName().equals(PPC.DEMOGRAPHY_MULTIPLIER_VALUES.getName())) {
            relatedId = Util.fieldAsLong(rowData, PPC.DEMOGRAPHY_MULTIPLIER_VALUES.DEMOGRAPHY_MULTIPLIER_VALUE_ID);
        } else if (row.getTableName().equals(PPC.GEO_MULTIPLIER_VALUES.getName())) {
            relatedId = Util.fieldAsLong(rowData, PPC.GEO_MULTIPLIER_VALUES.GEO_MULTIPLIER_VALUE_ID);
        } else if (row.getTableName().equals(PPC.RETARGETING_MULTIPLIER_VALUES.getName())) {
            relatedId = Util.fieldAsLong(rowData, PPC.RETARGETING_MULTIPLIER_VALUES.RETARGETING_MULTIPLIER_VALUE_ID);
        } else {
            throw new UnsupportedOperationException(row.getTableName());
        }
        return relatedId;
    }

    /**
     * Может вызываться только с кортежем из таблицы hierarchical_multipliers
     */
    private static ObjectPath getObjectPath(MySQLSimpleRowIndexed rowData, DictResponsesAccessor dictData) {
        ObjectPath path = dictData.getCampaignPath(Util.fieldAsLong(rowData, PPC.HIERARCHICAL_MULTIPLIERS.CID));
        Long adGroupId = rowData.getByNameNullable(PPC.HIERARCHICAL_MULTIPLIERS.PID.getName()).getValueAsLong();
        if (adGroupId == null) {
            return path;
        } else {
            return new ObjectPath.AdGroupPath((ObjectPath.CampaignPath) path, new AdGroupId(adGroupId));
        }
    }

    private static Pair<HierarchicalMultiplierRowModel, OnInsertAction> newDataForInsert(EnrichedInsertRow row,
                                                                                         DictResponsesAccessor dictData) {
        MySQLSimpleRowIndexed rowData = row.getFields();
        long rootId = Util.fieldAsLong(rowData, PPC.HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID);
        if (isRootTable(row)) {
            return Pair.of(
                    new HierarchicalMultiplierRowModel(new HierarchicalMultipliersData.Root()
                            .update(rowData)
                            .withPath(getObjectPath(rowData, dictData)),
                            getHierarchicalMultiplierId(rowData)),
                    rootHasRelatedTable(rowData)
                            ? OnInsertAction.WRITE_INTO_DICT_ONLY
                            : OnInsertAction.WRITE_INTO_LOG_AND_DICT_AS_INSERT);
        } else {
            long relatedId = getRelatedId(row, rowData);
            HierarchicalMultipliersData related;
            if (row.getTableName().equals(PPC.DEMOGRAPHY_MULTIPLIER_VALUES.getName())) {
                related = new HierarchicalMultipliersData.Demography();
            } else if (row.getTableName().equals(PPC.GEO_MULTIPLIER_VALUES.getName())) {
                related = new HierarchicalMultipliersData.Geo();
            } else if (row.getTableName().equals(PPC.RETARGETING_MULTIPLIER_VALUES.getName())) {
                related = new HierarchicalMultipliersData.Retargeting();
            } else {
                throw new UnsupportedOperationException(row.getTableName());
            }

            related.update(rowData);
            HierarchicalMultiplierRowModel result = new HierarchicalMultiplierRowModel(
                    HierarchicalMultipliersData.fromDictValue((String) dictData.get(
                            DictDataCategory.HIERARCHICAL_MULTIPLIERS_RECORD, rootId)),
                    getHierarchicalMultiplierId(rowData));

            HierarchicalMultipliersData.Root root = result.getRoot();
            boolean wasEmpty = root.relatedIsEmpty();
            root.putRelated(relatedId, related);
            // Всегда записывается самый свежий last_change, благо он есть в каждой таблице
            root.withLastChange(
                    rowData.getByName(PPC.HIERARCHICAL_MULTIPLIERS.LAST_CHANGE.getName()).getValueAsString());
            if (related instanceof HierarchicalMultipliersData.Retargeting) {
                ((HierarchicalMultipliersData.Retargeting) related).withRetCondName(
                        dictData.getRetargetingConditionName(
                                Util.fieldAsLong(rowData, PPC.RETARGETING_MULTIPLIER_VALUES.RET_COND_ID)));
            }
            return Pair.of(result, wasEmpty
                    ? OnInsertAction.WRITE_INTO_LOG_AND_DICT_AS_INSERT
                    : OnInsertAction.WRITE_INTO_LOG_AND_DICT_AS_UPDATE);
        }
    }

    /**
     * При определённых условиях вставка в mysql может быть записана в лог как изменение записи. Также удаление в mysql
     * может быть записано в лог как изменение. Поэтому существует три метода с одинаковым названием для трёх разных
     * источников входных данных.
     */
    private static HierarchicalMultiplierRowModel oldDataForUpdate(EnrichedInsertRow row,
                                                                   DictResponsesAccessor dictData) {
        MySQLSimpleRowIndexed after = row.getFields();
        long id = Util.fieldAsLong(after, PPC.HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID);
        return new HierarchicalMultiplierRowModel(HierarchicalMultipliersData.fromDictValue(
                (String) dictData.get(DictDataCategory.HIERARCHICAL_MULTIPLIERS_RECORD, id)),
                getHierarchicalMultiplierId(after));
    }

    /**
     * См. {@link #oldDataForUpdate(EnrichedInsertRow, DictResponsesAccessor)}
     */
    private static HierarchicalMultiplierRowModel oldDataForUpdate(EnrichedUpdateRow row,
                                                                   DictResponsesAccessor dictData) {
        MySQLSimpleRowIndexed before = row.getFields().getBefore();
        long id = Util.fieldAsLong(before, PPC.HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID);
        return new HierarchicalMultiplierRowModel(HierarchicalMultipliersData.fromDictValue(
                (String) dictData.get(DictDataCategory.HIERARCHICAL_MULTIPLIERS_RECORD, id)),
                getHierarchicalMultiplierId(before));
    }

    /**
     * См. {@link #oldDataForUpdate(EnrichedInsertRow, DictResponsesAccessor)}
     */
    private static HierarchicalMultiplierRowModel oldDataForUpdate(EnrichedDeleteRow row,
                                                                   DictResponsesAccessor dictData) {
        MySQLSimpleRowIndexed before = row.getFields();
        long id = Util.fieldAsLong(before, PPC.HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID);
        return new HierarchicalMultiplierRowModel(HierarchicalMultipliersData.fromDictValue(
                (String) dictData.get(DictDataCategory.HIERARCHICAL_MULTIPLIERS_RECORD, id)),
                getHierarchicalMultiplierId(before));
    }

    private static Optional<HierarchicalMultiplierRowModel> newDataForUpdate(EnrichedUpdateRow row,
                                                                             DictResponsesAccessor dictData) {
        MySQLSimpleRowIndexed after = row.getFields().getAfter();
        HierarchicalMultiplierRowModel result = oldDataForUpdate(row, dictData);
        HierarchicalMultipliersData.Root root = result.getRoot();
        if (isRootTable(row)) {
            root.update(after);
            return Optional.of(result);
        } else {
            long relatedId = getRelatedId(row, after);
            HierarchicalMultipliersData related = root.getRelated(relatedId);
            if (related == null) {
                // Стреляло в DIRECT-76704.
                // Когда-то было решено в DIRECT-76379, но пришлось отказаться: DIRECT-76785.
                // При инициализации словаря возможны гонки. На стадии заполнения словаря SQL-запросами
                // эта корректировка могла быть упущенна из-за того, что была удалена в неподходящий момент.
                // На стадии повтора бинлога обращение к этой корректировке вызывало ошибку.
                // При этом в обычном режиме записи логов такой ошибки быть не должно.
                logger.warn("hierarchical multipliers group with id {} does not contain multiplier with id {}."
                                + " Campaign id {}. Skipping this change",

                        relatedId,
                        result.getCampaignId().toLong());
                return Optional.empty();
            } else {
                related.update(after);
                // Всегда записывается самый свежий last_change, благо он есть в каждой таблице
                root.withLastChange(
                        after.getByName(PPC.HIERARCHICAL_MULTIPLIERS.LAST_CHANGE.getName()).getValueAsString());
                if (related instanceof HierarchicalMultipliersData.Retargeting) {
                    ((HierarchicalMultipliersData.Retargeting) related)
                            .withRetCondName(dictData.getRetargetingConditionName(
                                    Util.fieldAsLong(after, PPC.RETARGETING_MULTIPLIER_VALUES.RET_COND_ID)));
                }
                return Optional.of(result);
            }
        }
    }

    /**
     * То, что должно быть записано в словарь при удалении записи из mysql
     */
    private static Optional<HierarchicalMultiplierRowModel> newDictDataForDelete(EnrichedDeleteRow row,
                                                                                 DictResponsesAccessor dictData) {
        MySQLSimpleRowIndexed rowData = row.getFields();
        if (isRootTable(row)) {
            return Optional.empty();
        } else {
            long rootId = Util.fieldAsLong(rowData, PPC.HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID);
            HierarchicalMultiplierRowModel result = new HierarchicalMultiplierRowModel(
                    HierarchicalMultipliersData.fromDictValue((String) dictData.get(
                            DictDataCategory.HIERARCHICAL_MULTIPLIERS_RECORD, rootId)),
                    getHierarchicalMultiplierId(rowData));
            long relatedId = getRelatedId(row, rowData);
            result.getRoot().removeRelated(relatedId);
            return Optional.of(result);
        }
    }

    /**
     * То, что должно быть записано в логи при удалении записи из mysql, если должно
     */
    @Nullable
    private static Pair<HierarchicalMultiplierRowModel, OnDeleteAction> logDataForDelete(
            EnrichedDeleteRow row, DictResponsesAccessor dictData) {
        MySQLSimpleRowIndexed rowData = row.getFields();
        long rootId = Util.fieldAsLong(rowData, PPC.HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID);
        if (isRootTable(row)) {
            if (rootHasRelatedTable(rowData)) {
                return null;
            } else {
                HierarchicalMultiplierRowModel result = new HierarchicalMultiplierRowModel(
                        HierarchicalMultipliersData.fromDictValue((String) dictData.get(
                                DictDataCategory.HIERARCHICAL_MULTIPLIERS_RECORD, rootId)),
                        getHierarchicalMultiplierId(rowData));
                return Pair.of(result, OnDeleteAction.WRITE_INTO_BEFORE_AS_DELETE);
            }
        } else {
            HierarchicalMultiplierRowModel result = new HierarchicalMultiplierRowModel(
                    HierarchicalMultipliersData.fromDictValue((String) dictData.get(
                            DictDataCategory.HIERARCHICAL_MULTIPLIERS_RECORD, rootId)),
                    getHierarchicalMultiplierId(rowData));
            long relatedId = getRelatedId(row, rowData);
            HierarchicalMultipliersData.Root root = result.getRoot();
            if (root.relatedSize() <= 1 && root.relatedContainsKey(relatedId)) {
                return Pair.of(result, OnDeleteAction.WRITE_INTO_BEFORE_AS_DELETE);
            } else {
                root.removeRelated(relatedId);
                return Pair.of(result, OnDeleteAction.WRITE_INTO_AFTER_AS_UPDATE);
            }
        }
    }

    @Nullable
    private static Triple<Operation, ObjectPath, RowModelPair> processInsertEvent(
            EnrichedInsertRow row, DictResponsesAccessor dictResponsesAccessor) {
        Pair<HierarchicalMultiplierRowModel, OnInsertAction> decision = newDataForInsert(row, dictResponsesAccessor);
        HierarchicalMultiplierRowModel newData = decision.getLeft();
        OnInsertAction action = decision.getRight();
        switch (action) {
            case WRITE_INTO_LOG_AND_DICT_AS_INSERT:
                return Triple.of(
                        Operation.INSERT,
                        newData.getRoot().getPath(),
                        new RowModelPair<>(
                                new HierarchicalMultiplierRowModel(),
                                newData.convertToMap()));
            case WRITE_INTO_LOG_AND_DICT_AS_UPDATE:
                HierarchicalMultiplierRowModel oldData = oldDataForUpdate(row, dictResponsesAccessor);
                return Triple.of(
                        Operation.UPDATE,
                        oldData.getRoot().getPath(),
                        new RowModelPair<>(
                                oldData.convertToMap(),
                                newData.convertToMap()));
            case WRITE_INTO_DICT_ONLY:
                return null;
            default:
                throw new UnsupportedOperationException(action.name());
        }
    }

    private static Optional<Triple<Operation, ObjectPath, RowModelPair>> processUpdateEvent(
            EnrichedUpdateRow row, DictResponsesAccessor dictResponsesAccessor) {
        HierarchicalMultiplierRowModel oldData = oldDataForUpdate(row, dictResponsesAccessor);
        Optional<HierarchicalMultiplierRowModel> newData = newDataForUpdate(row, dictResponsesAccessor);
        if (newData.isPresent()) {
            if (!oldData.getRoot().getPath().equals(newData.get().getRoot().getPath())) {
                throw new IllegalStateException(String.format("Path changed from %s to %s",
                        oldData.getRoot().getPath(), newData.get().getRoot().getPath()));
            }
            return Optional.of(Triple.of(
                    Operation.UPDATE,
                    oldData.getRoot().getPath(),
                    new RowModelPair<>(
                            oldData.convertToMap(),
                            newData.get().convertToMap())));
        } else {
            return Optional.empty();
        }
    }

    @Nullable
    private static Triple<Operation, ObjectPath, RowModelPair> processDeleteEvent(
            EnrichedDeleteRow row, DictResponsesAccessor dictResponsesAccessor) {
        Pair<HierarchicalMultiplierRowModel, OnDeleteAction> decision =
                logDataForDelete(row, dictResponsesAccessor);
        if (decision != null) {
            HierarchicalMultiplierRowModel data = decision.getLeft();
            OnDeleteAction action = decision.getRight();
            switch (action) {
                case WRITE_INTO_AFTER_AS_UPDATE:
                    HierarchicalMultiplierRowModel oldData = oldDataForUpdate(row, dictResponsesAccessor);
                    return Triple.of(
                            Operation.UPDATE,
                            oldData.getRoot().getPath(),
                            new RowModelPair<>(
                                    oldData.convertToMap(),
                                    data.convertToMap()));
                case WRITE_INTO_BEFORE_AS_DELETE:
                    return Triple.of(
                            Operation.DELETE,
                            data.getRoot().getPath(),
                            new RowModelPair<>(
                                    data.convertToMap(),
                                    new HierarchicalMultiplierRowModel()));
                default:
                    throw new UnsupportedOperationException(action.toString());
            }
        } else {
            return null;
        }
    }

    @Override
    public void fillDictRequests(EnrichedRow row, DictRequestsFiller dictRequests) {
        if (rowCanBeHandled(row)) {
            // Если вставляется новый hierarchical_multipliers, то надо проверить его тип.
            // Для типов, связанных с *_multiplier_values нужны строки из соответствующих таблиц. Эти
            // строки ещё не были созданы, так что на текущий момент нет смысла записывать пустую группу в лог. А вот
            // записать её в словарь смысл есть. И тогда при последующем создании *_multiplier_values можно будет
            // записать в лог событие создания группы.
            MySQLSimpleRowIndexed data = Util.dataForGettingId(row);
            boolean needOldRecord;
            if (isRootTable(row)) {
                dictRequests.requireCampaignPath(Util.fieldAsLong(data, PPC.HIERARCHICAL_MULTIPLIERS.CID));
                needOldRecord = !(row instanceof EnrichedInsertRow);
            } else {
                needOldRecord = true;
                if (row.getTableName().equals(PPC.RETARGETING_MULTIPLIER_VALUES.getName())) {
                    dictRequests.requireRetargetingConditionName(Util.fieldAsLong(
                            data, PPC.RETARGETING_MULTIPLIER_VALUES.RET_COND_ID));
                }
            }
            if (needOldRecord) {
                long rootId = Util.fieldAsLong(data, PPC.HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID);
                dictRequests.require(DictDataCategory.HIERARCHICAL_MULTIPLIERS_RECORD, rootId);
            }
        }
    }

    @Nonnull
    @Override
    @SuppressWarnings("unchecked")
    public List<ActionLogRecord> processEvent(EnrichedRow row, DictResponsesAccessor dictResponsesAccessor) {
        if (rowCanBeHandled(row)) {
            Triple<Operation, ObjectPath, RowModelPair> data;
            if (row instanceof EnrichedInsertRow) {
                data = processInsertEvent((EnrichedInsertRow) row, dictResponsesAccessor);
            } else if (row instanceof EnrichedUpdateRow) {
                data = processUpdateEvent((EnrichedUpdateRow) row, dictResponsesAccessor).orElse(null);
            } else {
                data = processDeleteEvent((EnrichedDeleteRow) row, dictResponsesAccessor);
            }
            if (data != null) {
                Operation operation = data.getLeft();
                ObjectPath path = data.getMiddle();
                RowModelPair pair = data.getRight();
                pair.validate();
                fieldStrategy.handle(operation, pair, dictResponsesAccessor);
                pair.validate();
                return Collections.singletonList(ActionLogRecord.builder()
                        .withDateTime(row.getDateTime())
                        .withDb(row.getDbName())
                        .withDirectTraceInfo(row.getDirectTraceInfo())
                        .withGtid(row.getGtid())
                        .withQuerySerial(row.getQuerySerial())
                        .withRowSerial(row.getRowSerial())
                        .withType(ACTION_LOG_RECORD_TYPE)
                        .withNewFields(pair.after.toFieldValueList())
                        .withOldFields(pair.before.toFieldValueList())
                        .withOperation(operation)
                        .withPath(path)
                        .withRecordSource(recordSource)
                        .build());
            }
        }
        return Collections.emptyList();
    }

    @Override
    public void fillFreshDictValues(EnrichedRow row, DictResponsesAccessor dictData,
                                    FreshDictValuesFiller freshDictValues) {
        if (rowCanBeHandled(row)) {
            HierarchicalMultiplierRowModel data;
            if (row instanceof EnrichedInsertRow) {
                data = newDataForInsert((EnrichedInsertRow) row, dictData).getLeft();
            } else if (row instanceof EnrichedUpdateRow) {
                data = newDataForUpdate((EnrichedUpdateRow) row, dictData).orElse(null);
            } else {
                data = newDictDataForDelete((EnrichedDeleteRow) row, dictData).orElse(null);
            }
            if (data != null) {
                freshDictValues.add(
                        DictDataCategory.HIERARCHICAL_MULTIPLIERS_RECORD,
                        Util.fieldAsLong(Util.dataForGettingId(row),
                                PPC.HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID),
                        HierarchicalMultipliersData.toDictValue(data.getRoot()));
            }
        }
    }

    @Override
    public Collection<DictDataCategory> provides() {
        return Collections.singletonList(
                DictDataCategory.HIERARCHICAL_MULTIPLIERS_RECORD);
    }

    @Override
    public DictFiller makePureDictFiller() {
        // В словарь пишется то же самое, что и в лог, только в другом формате.
        return this;
    }

    private enum OnInsertAction {
        WRITE_INTO_LOG_AND_DICT_AS_INSERT,
        WRITE_INTO_LOG_AND_DICT_AS_UPDATE,
        WRITE_INTO_DICT_ONLY;
    }

    private enum OnDeleteAction {
        WRITE_INTO_AFTER_AS_UPDATE,
        WRITE_INTO_BEFORE_AS_DELETE;
    }
}
