package ru.yandex.direct.useractionlog.writer.generator;

import java.util.Collection;
import java.util.Collections;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;

import ru.yandex.direct.binlog.reader.EnrichedRow;
import ru.yandex.direct.binlog.reader.EnrichedUpdateRow;
import ru.yandex.direct.binlog.reader.MySQLSimpleRowIndexed;
import ru.yandex.direct.useractionlog.dict.DictDataCategory;
import ru.yandex.direct.useractionlog.dict.DictRequestsFiller;
import ru.yandex.direct.useractionlog.dict.DictResponsesAccessor;
import ru.yandex.direct.useractionlog.dict.FreshDictValuesFiller;
import ru.yandex.direct.useractionlog.model.RowModelPair;
import ru.yandex.direct.useractionlog.schema.ActionLogRecord;
import ru.yandex.direct.useractionlog.schema.ObjectPath;
import ru.yandex.direct.useractionlog.schema.RecordSource;

/**
 * Генерация записей в таблицу пользовательских логов, которая составляет события путём комбинирования результатов
 * {@link ObjectPathStrategy} и {@link FieldsStrategy}.
 */
@ParametersAreNonnullByDefault
class ModularRowProcessingStrategy implements RowProcessingStrategy {
    private final RecordSource recordSource;
    @Nonnull
    private final ObjectPathStrategy objectPathStrategy;

    @Nonnull
    private final FieldsStrategy fieldsStrategy;

    ModularRowProcessingStrategy(RecordSource recordSource, ObjectPathStrategy objectPathStrategy,
                                 FieldsStrategy fieldsStrategy) {
        this.recordSource = recordSource;
        this.objectPathStrategy = objectPathStrategy;
        this.fieldsStrategy = fieldsStrategy;
    }

    private void checkPrimaryKeyNotChanged(EnrichedUpdateRow row) {
        MySQLSimpleRowIndexed before = row.getFields().getBefore();
        MySQLSimpleRowIndexed after = row.getFields().getAfter();
        for (String columnName : objectPathStrategy.objectPathColumns()) {
            if (!before.getByName(columnName).equals(after.getByName(columnName))) {
                throw new UnsupportedOperationException("Can't handle row with changing ObjectPath: " + row);
            }
        }
    }

    @Override
    public void fillDictRequests(EnrichedRow row, DictRequestsFiller dictRequests) {
        if (row instanceof EnrichedUpdateRow) {
            checkPrimaryKeyNotChanged((EnrichedUpdateRow) row);
        }
        objectPathStrategy.fillDictRequests(row, dictRequests);
        fieldsStrategy.fillDictRequests(row, dictRequests);
    }

    @Nonnull
    @Override
    public List<ActionLogRecord> processEvent(EnrichedRow row, DictResponsesAccessor dictResponsesAccessor) {
        RowModelPair rowModelPair = RowModelPair.fromRow(row);
        fieldsStrategy.handle(
                Util.operationFromRow(row),
                rowModelPair,
                dictResponsesAccessor);
        rowModelPair.validate();
        if (rowModelPair.isEmpty()) {
            return Collections.emptyList();
        } else {
            ObjectPath objectPath = objectPathStrategy.extract(row, dictResponsesAccessor);
            return Collections.singletonList(ActionLogRecord.builderFrom(row, recordSource)
                    .withPath(objectPath)
                    .withOldFields(rowModelPair.before.toFieldValueList())
                    .withNewFields(rowModelPair.after.toFieldValueList())
                    .build());
        }
    }

    @Override
    public DictFiller makePureDictFiller() {
        return new DictFillerChain(
                objectPathStrategy.makePureDictFiller(),
                fieldsStrategy.makePureDictFiller());
    }

    @Override
    public void fillFreshDictValues(EnrichedRow row, DictResponsesAccessor dictData,
                                    FreshDictValuesFiller freshDictValues) {
        objectPathStrategy.fillFreshDictValues(row, dictData, freshDictValues);
        fieldsStrategy.fillFreshDictValues(row, dictData, freshDictValues);
    }

    @Override
    public Collection<DictDataCategory> provides() {
        ImmutableSet.Builder<DictDataCategory> builder = ImmutableSet.builder();
        builder.addAll(objectPathStrategy.provides());
        builder.addAll(fieldsStrategy.provides());
        return builder.build();
    }
}
