package ru.yandex.direct.useractionlog.writer.generator;

import java.util.Collection;
import java.util.Collections;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;

import ru.yandex.direct.binlog.reader.EnrichedRow;
import ru.yandex.direct.useractionlog.dict.DictDataCategory;
import ru.yandex.direct.useractionlog.dict.DictRequestsFiller;
import ru.yandex.direct.useractionlog.dict.DictResponsesAccessor;
import ru.yandex.direct.useractionlog.dict.FreshDictValuesFiller;
import ru.yandex.direct.useractionlog.schema.ActionLogRecord;

/**
 * По названию таблицы выбирает один из {@link RowProcessingStrategy} и вызывает у него соответствующие методы.
 */
@ParametersAreNonnullByDefault
public class RowProcessingStrategyTableSwitch implements RowProcessingStrategy {
    private final ImmutableMap<String, RowProcessingStrategy> byTableName;

    /**
     * @param byTableName Соответствие названия таблицы и обработчика для этой таблицы.
     */
    public RowProcessingStrategyTableSwitch(ImmutableMap<String, RowProcessingStrategy> byTableName) {
        this.byTableName = byTableName;
    }

    @Override
    public void fillDictRequests(EnrichedRow row, DictRequestsFiller dictRequests) {
        RowProcessingStrategy chosen = byTableName.get(row.getTableName());
        if (chosen != null) {
            chosen.fillDictRequests(row, dictRequests);
        }
    }

    @Nonnull
    @Override
    public List<ActionLogRecord> processEvent(EnrichedRow row, DictResponsesAccessor dictResponsesAccessor) {
        RowProcessingStrategy chosen = byTableName.get(row.getTableName());
        if (chosen != null) {
            return chosen.processEvent(row, dictResponsesAccessor);
        } else {
            return Collections.emptyList();
        }
    }

    @Override
    public DictFillerTableSwitch makePureDictFiller() {
        ImmutableMap.Builder<String, DictFiller> builder = ImmutableMap.builder();
        byTableName.forEach((table, strategy) -> {
            DictFiller pureFiller = strategy.makePureDictFiller();
            if (!pureFiller.provides().isEmpty()) {
                builder.put(table, pureFiller);
            }
        });
        return new DictFillerTableSwitch(builder.build());
    }

    @Override
    public Collection<DictDataCategory> provides() {
        return Collections.emptyList();
    }

    @Override
    public void fillFreshDictValues(EnrichedRow row, DictResponsesAccessor dictData,
                                    FreshDictValuesFiller freshDictValues) {
        RowProcessingStrategy chosen = byTableName.get(row.getTableName());
        if (chosen != null) {
            chosen.fillFreshDictValues(row, dictData, freshDictValues);
        }
    }
}
