package ru.yandex.direct.useractionlog.writer.generator;

import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.direct.binlog.reader.EnrichedDeleteRow;
import ru.yandex.direct.binlog.reader.EnrichedInsertRow;
import ru.yandex.direct.binlog.reader.EnrichedRow;
import ru.yandex.direct.binlog.reader.EnrichedUpdateRow;
import ru.yandex.direct.binlog.reader.MySQLSimpleRowIndexed;
import ru.yandex.direct.mysql.MySQLColumnData;
import ru.yandex.direct.useractionlog.dict.DictDataCategory;
import ru.yandex.direct.useractionlog.dict.DictRequestsFiller;
import ru.yandex.direct.useractionlog.dict.DictResponsesAccessor;
import ru.yandex.direct.useractionlog.dict.FreshDictValuesFiller;
import ru.yandex.direct.useractionlog.model.RowModel;
import ru.yandex.direct.useractionlog.model.RowModelPair;

import static ru.yandex.direct.useractionlog.writer.generator.Util.getGroupsByFieldName;

/**
 * Стратегия, когда нужно взять из словаря какие-то тестовые поля и добавить их в строчку пользовательского лога как
 * старые данные.
 * <p>
 * Применимо для текстовых полей, у которых в БД указан тип text. В режиме репликации noblob при изменении такого поля
 * оно будет присутствовать в after, но не в before. При удалении записи это поле также будет отсутствовать в before.
 * В таких случаях эта стратегия добавляет желаемое поле в before.
 * <p>
 * <b>Важно:</b> Стратегия превращает null в пустые строки для подконтрольных ей полей. Так сделано потому, что словари
 * не умеют сохранять null, они преобразовывают их в пустые строки, и лучше везде всё преобразовывать в пустые строки,
 * чем преобразовывать некоторые поля непредсказуемо.
 */
@ParametersAreNonnullByDefault
class StringsFromDictStrategy implements FieldsStrategy, DictFiller, PureDictFillerFactory {
    private final String idField;
    private final Map<String, DictDataCategory> categoryByFieldName;
    private final ImmutableSet<DictDataCategory> canCreateDictValueFor;
    private final ImmutableSet<DictDataCategory> writeInUpdateOnly;
    private final ImmutableMap<String, ImmutableList<Group>> groupsByFieldName;

    private StringsFromDictStrategy(String idField, LinkedHashMap<String, DictDataCategory> categoryByFieldName,
                                    ImmutableSet<DictDataCategory> canCreateDictValueFor,
                                    ImmutableSet<DictDataCategory> writeInUpdateOnly,
                                    Collection<Group> groups) {
        this.idField = idField;
        this.categoryByFieldName = Collections.unmodifiableMap(categoryByFieldName);
        this.canCreateDictValueFor = canCreateDictValueFor;
        this.writeInUpdateOnly = writeInUpdateOnly;
        this.groupsByFieldName = getGroupsByFieldName(groups);
    }

    static Builder builder() {
        return new Builder();
    }

    @Override
    public void fillDictRequests(EnrichedRow row, DictRequestsFiller dictRequests) {
        if (row instanceof EnrichedUpdateRow) {
            MySQLSimpleRowIndexed after = ((EnrichedUpdateRow) row).getFields().getAfter();
            Map<String, String> beforeMap = ((EnrichedUpdateRow) row).getFields().getBefore().toMap();
            Map<String, String> afterMap = after.toMap();
            long id = Long.parseLong(beforeMap.get(idField));
            for (MySQLColumnData afterCell : after) {
                String name = afterCell.getSchema().getName();
                requireCategory(name, dictRequests, beforeMap, id);
                List<Group> groups = groupsByFieldName.getOrDefault(name, ImmutableList.of());
                groups.forEach(group -> requireCategoryForGroup(group, dictRequests, beforeMap, afterMap, id));
            }
        } else if (row instanceof EnrichedDeleteRow) {
            long id = ((Number) ((EnrichedDeleteRow) row).getFields()
                    .getByName(idField).getRawValue()).longValue();
            Set<String> requiredFieldNamesFromBefore = new LinkedHashSet<>(categoryByFieldName.keySet());
            for (MySQLColumnData cell : ((EnrichedDeleteRow) row).getFields()) {
                requiredFieldNamesFromBefore.remove(cell.getSchema().getName());
            }
            for (String name : requiredFieldNamesFromBefore) {
                dictRequests.require(categoryByFieldName.get(name), id);
            }
        }
    }

    /**
     * Если в группе какое-либо поле изменилось (даже не текстовое), то для всех текстовых полей из группы
     * (не важно менявшихся или не менявшихся), для которых нет предыдущего значения мы отправляем просьбу в
     * DictRequestsFiller подгрузить это значение.
     */
    private void requireCategoryForGroup(Group group,
                                         DictRequestsFiller dictRequests,
                                         Map<String, String> beforeMap,
                                         Map<String, String> afterMap,
                                         long id) {
        var shouldAddFieldsInBefore = !group.extract(beforeMap).equals(group.extract(afterMap));
        if (shouldAddFieldsInBefore) {
            for (var fieldName : group.getNames()) {
                requireCategory(fieldName, dictRequests, beforeMap, id);
            }
        }
    }

    /**
     * Если fieldName это текстовое поле, для которого нет предыдущего значения - отправляем просьбу в
     * DictRequestsFiller подгрузить это значение.
     */
    private void requireCategory(String fieldName,
                                 DictRequestsFiller dictRequestsFiller,
                                 Map<String, String> beforeMap,
                                 long id) {
        if (!beforeMap.containsKey(fieldName)) {
            DictDataCategory category = categoryByFieldName.get(fieldName);
            if (category != null) {
                dictRequestsFiller.require(category, id);
            }
        }
    }

    @Override
    public void handleInsert(RowModel after, DictResponsesAccessor dictResponsesAccessor) {
        nullsToEmptyString(after);
    }

    @Override
    public void handleUpdate(RowModelPair pair, DictResponsesAccessor dictResponsesAccessor) {
        Set<String> requiredFieldNamesFromBefore = new LinkedHashSet<>();
        Map<String, String> beforeMap = pair.before.getMap();
        Map<String, String> afterMap = pair.after.getMap();
        for (Map.Entry<String, String> entry : afterMap.entrySet()) {
            String name = entry.getKey();
            if (categoryByFieldName.containsKey(name)) {
                requiredFieldNamesFromBefore.add(name);
            }
            for (var group : groupsByFieldName.getOrDefault(name, ImmutableList.of())) {
                var shouldAddFieldsInBefore = !group.extract(beforeMap).equals(group.extract(afterMap));
                if (shouldAddFieldsInBefore) {
                    group.getNames().stream()
                            .filter(categoryByFieldName::containsKey)
                            .forEach(requiredFieldNamesFromBefore::add);
                }
            }
        }

        Long id = null;
        for (Map.Entry<String, String> entry : beforeMap.entrySet()) {
            String name = entry.getKey();
            requiredFieldNamesFromBefore.remove(name);
            if (name.equals(idField)) {
                id = Long.parseLong(entry.getValue());
            }
        }
        if (id == null) {
            throw new IllegalStateException("Not found: " + idField);
        }
        if (!requiredFieldNamesFromBefore.isEmpty()) {
            for (String name : requiredFieldNamesFromBefore) {
                beforeMap.put(name, (String) dictResponsesAccessor.get(categoryByFieldName.get(name), id));
            }
        }
        nullsToEmptyString(pair.before);
        nullsToEmptyString(pair.after);
        pair.validate();
    }

    @Override
    public void handleDelete(RowModel before, DictResponsesAccessor dictResponsesAccessor) {
        Set<String> requiredFieldNamesFromBefore = new LinkedHashSet<>(categoryByFieldName.keySet());
        requiredFieldNamesFromBefore.removeIf(field -> writeInUpdateOnly.contains(categoryByFieldName.get(field)));
        Long id = null;
        for (Map.Entry<String, String> entry : before.getMap().entrySet()) {
            String name = entry.getKey();
            requiredFieldNamesFromBefore.remove(name);
            if (name.equals(idField)) {
                id = Long.parseLong(entry.getValue());
            }
        }
        if (id == null) {
            throw new IllegalStateException("Not found: " + idField);
        }
        LinkedHashMap<String, String> beforeMap = before.getMap();
        if (!requiredFieldNamesFromBefore.isEmpty()) {
            for (String name : requiredFieldNamesFromBefore) {
                beforeMap.put(name, (String) dictResponsesAccessor.get(categoryByFieldName.get(name), id));
            }
        }
        nullsToEmptyString(before);
        before.validate();
    }

    @Override
    public void fillFreshDictValues(EnrichedRow row, DictResponsesAccessor dictData,
                                    FreshDictValuesFiller freshDictValues) {
        MySQLSimpleRowIndexed rowData;
        if (row instanceof EnrichedInsertRow) {
            rowData = ((EnrichedInsertRow) row).getFields();
        } else if (row instanceof EnrichedUpdateRow) {
            rowData = ((EnrichedUpdateRow) row).getFields().getAfter();
        } else {
            return;
        }
        long id = Util.fieldAsLong(rowData, idField);
        for (MySQLColumnData cell : rowData) {
            DictDataCategory category = categoryByFieldName.get(cell.getSchema().getName());
            if (category != null && canCreateDictValueFor.contains(category)) {
                freshDictValues.add(category, id, StringUtils.defaultString(cell.getValueAsString()));
            }
        }
    }

    private void nullsToEmptyString(RowModel model) {
        for (Map.Entry<String, String> entry : model.getMap().entrySet()) {
            if (entry.getValue() == null && categoryByFieldName.containsKey(entry.getKey())) {
                entry.setValue("");
            }
        }
    }

    @Override
    public DictFiller makePureDictFiller() {
        if (canCreateDictValueFor.isEmpty()) {
            return DummyFiller.INSTANCE;
        } else {
            return new DictFiller() {
                @Override
                public void fillDictRequests(EnrichedRow row, DictRequestsFiller dictRequests) {
                    // По умолчанию этот класс запрашивает словарные данные, чтобы вставить название в событие
                    // изменения/удаления объекта. Если стоит цель только сгенерировать новые словарные данные, то
                    // ничего запрашивать не надо.
                }

                @Override
                public void fillFreshDictValues(EnrichedRow row, DictResponsesAccessor dictData,
                                                FreshDictValuesFiller freshDictValues) {
                    StringsFromDictStrategy.this.fillFreshDictValues(row, dictData, freshDictValues);
                }

                @Override
                public Collection<DictDataCategory> provides() {
                    return StringsFromDictStrategy.this.provides();
                }
            };
        }
    }

    static class Builder {
        private String idField;
        private LinkedHashMap<String, DictDataCategory> categoryByFieldName = new LinkedHashMap<>();
        private Set<DictDataCategory> doesNotProduceDictValue = EnumSet.noneOf(DictDataCategory.class);
        private Set<DictDataCategory> writeInUpdateOnly = EnumSet.noneOf(DictDataCategory.class);
        private final ImmutableList.Builder<Group> groupsBuilder = ImmutableList.builder();

        private Builder() {
        }

        /**
         * @param idField Название колонки, в которой хранится идентификатор для словаря.
         */
        Builder withIdField(String idField) {
            this.idField = idField;
            return this;
        }

        /**
         * @param category  Категория словарных данных.
         * @param fieldName Название колонки, в которую будет записываться старое словарное значение и из которой будет
         *                  браться новое словарное значение.
         */
        Builder with(DictDataCategory category, String fieldName) {
            categoryByFieldName.forEach((existingFieldName, existingCategory) -> {
                if (existingCategory == category) {
                    throw new IllegalArgumentException(
                            "Category " + category + " already used for field " + existingFieldName);
                }
                if (existingFieldName.equals(fieldName)) {
                    throw new IllegalArgumentException(
                            "Field name " + fieldName + " already used for category " + existingCategory);
                }
            });
            categoryByFieldName.put(fieldName, category);
            return this;
        }

        /**
         * По умоланию поля, указанные в {@link #with(DictDataCategory, String)}, будут использоваться и для получения
         * словарных данных, и для генерации новых словарных данных. Если из интересуемой таблицы невозможно извлечь
         * новые словарные данные определённой категории, такую категорию следует указать в вызове этого метода.
         */
        Builder doesNotProduceDictValue(DictDataCategory category) {
            this.doesNotProduceDictValue.add(category);
            return this;
        }

        /**
         * По умолчанию поля, указанные в {@link #with(DictDataCategory, String)}, будут записываться в before
         * и при изменении строки в БД, и при удалении строки из БД. Если определённую категорию нужно писать только
         * при изменении, то такую категорию следует указать в вызове этого метода.
         */
        Builder writeInUpdateOnly(DictDataCategory category) {
            this.writeInUpdateOnly.add(category);
            return this;
        }

        /**
         * Объявляет группу полей. Если хотя бы одно из полей в группе изменилось, то все текстовые поля этой группы
         * отсутсвующие в before будут запрощены из словаря (не текстовые поля и так будут в before, так работает noblob
         * режим бинлогов в mysql).
         */
        Builder withFieldGroup(String... fields) {
            groupsBuilder.add(new GroupAsIs(ImmutableList.copyOf(fields)));
            return this;
        }

        StringsFromDictStrategy build() {
            return new StringsFromDictStrategy(
                    Objects.requireNonNull(idField),
                    categoryByFieldName,
                    ImmutableSet.copyOf(categoryByFieldName.values().stream()
                            .filter(category -> !doesNotProduceDictValue.contains(category))
                            .collect(Collectors.toSet())),
                    ImmutableSet.copyOf(writeInUpdateOnly),
                    groupsBuilder.build());
        }
    }

    @Override
    public Collection<DictDataCategory> provides() {
        return canCreateDictValueFor;
    }

}
