package ru.yandex.direct.useractionlog.writer.generator;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import org.jooq.TableField;

import ru.yandex.direct.binlog.reader.EnrichedDeleteRow;
import ru.yandex.direct.binlog.reader.EnrichedInsertRow;
import ru.yandex.direct.binlog.reader.EnrichedRow;
import ru.yandex.direct.binlog.reader.EnrichedUpdateRow;
import ru.yandex.direct.binlog.reader.MySQLSimpleRowIndexed;
import ru.yandex.direct.binlog.reader.MySQLUpdateRowIndexed;
import ru.yandex.direct.binlogclickhouse.schema.FieldValue;
import ru.yandex.direct.binlogclickhouse.schema.FieldValueList;
import ru.yandex.direct.mysql.MySQLColumnData;
import ru.yandex.direct.useractionlog.schema.Operation;

/**
 * Вспомогательные функции, применимые только в рамках генерации записей в таблицу пользовательских логов.
 */
@ParametersAreNonnullByDefault
class Util {
    private Util() {
        throw new UnsupportedOperationException("static members only");
    }

    /**
     * Для события создания записи в таблице возвращает кортеж после её создания, для изменения записи в таблице
     * возвращает кортеж до изменения записи, для удаления записи в таблице возвращает кортеж до удаления записи.
     *
     * @param row Событие о создании/измененении/удалении записи.
     * @return Набор значений для извлечения данных, необходимых для создания
     * {@link ru.yandex.direct.useractionlog.schema.ObjectPath}.
     */
    @Nonnull
    static MySQLSimpleRowIndexed dataForGettingId(EnrichedRow row) {
        if (row instanceof EnrichedUpdateRow) {
            return ((EnrichedUpdateRow) row).getFields().getBefore();
        } else if (row instanceof EnrichedInsertRow) {
            return ((EnrichedInsertRow) row).getFields();
        } else if (row instanceof EnrichedDeleteRow) {
            return ((EnrichedDeleteRow) row).getFields();
        } else {
            throw new UnsupportedOperationException("Unknown type of row " + row);
        }
    }

    /**
     * Извлечение кортежа с данными, в котором могут быть свежие словарные данные.
     *
     * @return Кортеж или null, если на вход пришло событие об удалении строки.
     */
    @Nullable
    static MySQLSimpleRowIndexed dataForFillingDict(EnrichedRow row) {
        if (row instanceof EnrichedInsertRow) {
            return ((EnrichedInsertRow) row).getFields();
        } else if (row instanceof EnrichedUpdateRow) {
            return ((EnrichedUpdateRow) row).getFields().getAfter();
        } else {
            return null;
        }
    }

    /**
     * Список всех кортежей, которые есть в событии изменения строки. При INSERT и DELETE в событии всегда один кортеж,
     * при UPDATE всегда два.
     */
    static Collection<MySQLSimpleRowIndexed> extractAllIndexedRows(EnrichedRow row) {
        if (row instanceof EnrichedUpdateRow) {
            MySQLUpdateRowIndexed fields = ((EnrichedUpdateRow) row).getFields();
            return Arrays.asList(fields.getBefore(), fields.getAfter());
        } else if (row instanceof EnrichedInsertRow) {
            return Collections.singleton(((EnrichedInsertRow) row).getFields());
        } else if (row instanceof EnrichedDeleteRow) {
            return Collections.singleton(((EnrichedDeleteRow) row).getFields());
        } else {
            throw new UnsupportedOperationException("Unknown type of row " + row);
        }
    }

    /**
     * Получить long из значения колонки
     */
    static long fieldAsLong(MySQLSimpleRowIndexed data, TableField field) {
        return fieldAsLong(data, field.getName());
    }

    /**
     * Получить long из значения колонки
     */
    static long fieldAsLong(MySQLSimpleRowIndexed data, String field) {
        MySQLColumnData fieldValue = data.getByNameNullable(field);
        Objects.requireNonNull(fieldValue, "No field with name " + field);
        return Long.parseLong(fieldValue.getValueAsString());
    }

    /**
     * Получить long из значения колонки
     */
    static long fieldAsLong(FieldValueList data, String field) {
        for (FieldValue<String> fieldValue : data.getFieldsValues()) {
            if (fieldValue.getName().equals(field)) {
                return Long.parseLong(fieldValue.getValue());
            }
        }
        throw new IllegalArgumentException("No field with name " + field);
    }

    /**
     * Подставляет словарное значение в список полей. Если колонка назначения уже есть в списке, её значение изменяется.
     * Если нету - добавляется новая в конец. Порядок следования колонок сохраняется.
     *
     * @param fieldValueList Список колонок.
     * @param dictGetter     Функция, берущая на вход идентификатор словарного объекта и возвращающая словарное
     *                       значение.
     * @param idField        Название колонки, в которой хранится идентификатор словарного объекта.
     * @param nameField      Название колонки, в которой следует записать словарное значение.
     * @return Новый список колонок.
     */
    static FieldValueList addToFieldValueList(FieldValueList fieldValueList,
                                              Function<Long, String> dictGetter,
                                              String idField,
                                              String nameField) {
        Optional<Long> id = Optional.empty();  // Может быть каким угодно значением
        int replaceIndex = -1;  // Индекс в массиве не может быть отрицательным
        int index = 0;
        for (FieldValue<String> candidate : fieldValueList.getFieldsValues()) {
            if (candidate.getName().equals(idField)) {
                id = Optional.of(Long.parseLong(candidate.getValue()));
            } else if (candidate.getName().equals(nameField)) {
                replaceIndex = index;
            }
            ++index;
        }
        if (!id.isPresent()) {
            throw new IllegalStateException("Field " + idField + " was not found in field list");
        }
        List<FieldValue<String>> result = new ArrayList<>(fieldValueList.getFieldsValues().size() + 1);
        result.addAll(fieldValueList.getFieldsValues());
        FieldValue<String> newField = new FieldValue<>(nameField, dictGetter.apply(id.get()));
        if (replaceIndex == -1) {
            result.add(newField);
        } else {
            result.set(replaceIndex, newField);
        }
        return new FieldValueList(result);
    }

    static Operation operationFromRow(EnrichedRow row) {
        if (row instanceof EnrichedInsertRow) {
            return Operation.INSERT;
        } else if (row instanceof EnrichedUpdateRow) {
            return Operation.UPDATE;
        } else if (row instanceof EnrichedDeleteRow) {
            return Operation.DELETE;
        } else {
            throw new UnsupportedOperationException(row.getClass().toString());
        }
    }

    static ImmutableMap<String, ImmutableList<Group>> getGroupsByFieldName(Collection<Group> groups) {
        Map<String, ImmutableList.Builder<Group>> groupByFieldNameTemp = new HashMap<>();
        for (Group group : groups) {
            for (String fieldName : group.getNames()) {
                groupByFieldNameTemp.computeIfAbsent(fieldName, k -> ImmutableList.builder()).add(group);
            }
        }
        ImmutableMap.Builder<String, ImmutableList<Group>> groupByFieldNameBuilder = ImmutableMap.builder();
        groupByFieldNameTemp.forEach((fieldName, groupListBuilder) ->
                groupByFieldNameBuilder.put(fieldName, groupListBuilder.build()));
        return groupByFieldNameBuilder.build();
    }

}
