package ru.yandex.direct.useractionlog.reader;

import java.sql.Date;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Deque;
import java.util.Iterator;
import java.util.List;
import java.util.function.Consumer;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.clickhouse.SqlBuilder;
import ru.yandex.direct.useractionlog.db.ReadActionLogTable;
import ru.yandex.direct.useractionlog.schema.ActionLogRecord;
import ru.yandex.direct.useractionlog.schema.ActionLogSchema;

/**
 * Последовательное считывание таблицы пользовательских логов SQL-запросами, фильтрующими по набору дат.
 * <p>
 * Первым запросом приблизительно* выясняет, сколько записей в таблице логов для каждого дня.
 * Затем в порядке убывания группирует дни так, чтобы внутри одной группы было запрошено минимум
 * {@code tryFetchAtLeast} записей. Затем последовательными запросами считывает записи из таблицы, применяя
 * каждую группу дат как фильтр.
 * <p>
 * Пример:
 * <ol>
 * <li><code>SELECT date, count() FROM user_action_log WHERE ... GROUP BY date</code>
 * Предположим, результат такой:
 * <ul>
 * <li>1970-01-01 - 10 записей</li>
 * <li>1970-01-02 - 10 записей</li>
 * <li>1970-01-03 - 10 записей</li>
 * </ul>
 * </li>
 * <li>Предположим, {@code tryFetchAtLeast=11}. Тогда первая группа - 1970-01-03 и 1970-01-02, с этим набором дат
 * получается получить 20 записей. Вторая группа - последняя дата 1970-01-01.</li>
 * <li><code>SELECT * FROM user_action_log WHERE date IN ('1970-01-03', '1970-01-02') AND ...</code></li>
 * <li><code>SELECT * FROM user_action_log WHERE date IN ('1970-01-01') AND ...</code></li>
 * </ol>
 * <p>
 * [*] Приблизительно - потому, что в таблице CollapsingMergeTree может быть несколько записей с одинаковым
 * первичным ключом. Можно считывать только самые свежие записи для каждого первичного ключа, но это
 * требует в разы больше времени.
 */
@ParametersAreNonnullByDefault
class DateSplittingActionLogReader implements Iterator<ActionLogRecord> {
    private final long tryFetchAtLeast;
    private final ActionLogRecordIteratorFactory actionLogRecordIteratorFactory;
    private final Deque<ReadActionLogTable.DateAndCount> dateAndCountStack;
    private final ReadActionLogTable readActionLogTable;
    private final Consumer<SqlBuilder> applyFilter;
    private final ReadRequestStats readRequestStats;
    @Nullable
    private final ReadActionLogTable.Offset offset;
    private final ReadActionLogTable.Order order;
    private boolean didNotFetchAnyChunk;
    private Iterator<ActionLogRecord> currentChunkIterator;

    /**
     * @param actionLogRecordIteratorFactory Некая функция, которая способна считать записи из таблицы пользовательских
     *                                       логов с применением фильтра {@code applyFilter}, смещения {@code offset} и
     *                                       порядка {@code order}. Эти фильтр, смещение и порядок будут передаваться
     *                                       функии через аргументы.
     * @param applyFilter                    Обработчик, который накладывает необходимые фильтры на SqlBuilder. Этот
     *                                       фильтр будет применяться и при подсчёте количества записей для каждого дня,
     *                                       и при непосредственном чтении записей.
     * @param tryFetchAtLeast                По возможности группировать даты так, чтобы при запросе логов для этих дат
     *                                       было получено записей не меньше указанного.
     * @param offset                         Смещение в таблице пользовательских логов. Для постраничного доступа.
     * @param order                          Порядок сортировки по дате и времени: по возрастанию или по убыванию.
     * @param readRequestStats               Объект, в который будет будет помещена статистика о проведённых
     */
    DateSplittingActionLogReader(
            ReadActionLogTable readActionLogTable,
            ActionLogRecordIteratorFactory actionLogRecordIteratorFactory,
            Consumer<SqlBuilder> applyFilter,
            long tryFetchAtLeast,
            @Nullable ReadActionLogTable.Offset offset,
            ReadActionLogTable.Order order,
            ReadRequestStats readRequestStats) {
        this.tryFetchAtLeast = tryFetchAtLeast;
        this.actionLogRecordIteratorFactory = actionLogRecordIteratorFactory;
        this.readActionLogTable = readActionLogTable;
        this.applyFilter = applyFilter;
        this.order = order;
        this.readRequestStats = readRequestStats;
        this.offset = offset;
        this.dateAndCountStack = new ArrayDeque<>();
        this.didNotFetchAnyChunk = true;
        this.currentChunkIterator = Collections.emptyIterator();
    }

    @Override
    public boolean hasNext() {
        tryPopulateStacks();
        return currentChunkIterator.hasNext() || !dateAndCountStack.isEmpty();
    }

    @Override
    public ActionLogRecord next() {
        tryPopulateStacks();
        return currentChunkIterator.next();
    }

    private void tryPopulateStacks() {
        if (!currentChunkIterator.hasNext()) {
            if (didNotFetchAnyChunk) {
                populateDateAndCountStack();
            }
            Collection<Date> datesForQuery = new ArrayList<>();
            if (didNotFetchAnyChunk && offset != null && !dateAndCountStack.isEmpty()) {
                // В запросе есть смещение. Так как смещение содержит дату, то есть смысл применять смещение
                // только к одному, самому новому дню. Для остальных запросов смещение не вырежет ни одной записи.
                datesForQuery.add(java.sql.Date.valueOf(dateAndCountStack.pop().getDate()));
            } else {
                long expectedRecords = 0;
                while (expectedRecords < tryFetchAtLeast && !dateAndCountStack.isEmpty()) {
                    ReadActionLogTable.DateAndCount dateAndCount = dateAndCountStack.pop();
                    datesForQuery.add(java.sql.Date.valueOf(dateAndCount.getDate()));
                    expectedRecords += dateAndCount.getCount();
                }
            }
            populateRecordStack(datesForQuery);
        }
    }

    private void populateDateAndCountStack() {
        SqlBuilder sqlBuilder = readActionLogTable.dateAndCountSqlBuilder(offset, order);
        applyFilter.accept(sqlBuilder);
        List<ReadActionLogTable.DateAndCount> dateAndCountList = readActionLogTable.getCountByDate(sqlBuilder);
        Comparator<ReadActionLogTable.DateAndCount> dateComparator =
                Comparator.comparing(ReadActionLogTable.DateAndCount::getDate);
        if (order == ReadActionLogTable.Order.DESC) {
            dateComparator = dateComparator.reversed();
        }
        dateAndCountList.sort(dateComparator);
        dateAndCountStack.addAll(dateAndCountList);
        ++readRequestStats.dateCountQueriesDone;
    }

    private void populateRecordStack(Collection<Date> datesForQuery) {
        if (datesForQuery.isEmpty()) {
            currentChunkIterator = Collections.emptyIterator();
        } else {
            currentChunkIterator = actionLogRecordIteratorFactory.apply(
                    sqlBuilder -> {
                        applyFilter.accept(sqlBuilder);
                        sqlBuilder.whereIn(new SqlBuilder.Column(ActionLogSchema.DATE.getName()), datesForQuery);
                    },
                    // Как уже написано выше, смещение имеет смысл применять только к самому первому запросу.
                    // Для остальных запросов смещение ничего не вырежет.
                    didNotFetchAnyChunk ? offset : null,
                    order);
            didNotFetchAnyChunk = false;
        }
    }
}
