package ru.yandex.direct.useractionlog.reader;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalLong;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterators;
import com.google.common.collect.PeekingIterator;

import ru.yandex.direct.clickhouse.SqlBuilder;
import ru.yandex.direct.useractionlog.db.ReadPpclogApiTable;
import ru.yandex.direct.useractionlog.schema.ActionLogRecord;
import ru.yandex.direct.useractionlog.schema.ActionLogRecordWithStats;
import ru.yandex.direct.useractionlog.schema.PpclogApiRecord;
import ru.yandex.direct.useractionlog.schema.RecordStats;

import static ru.yandex.direct.useractionlog.Services.API_SERVICES;

/**
 * Итератор, определяющий источник изменения {@link ActionLogRecord}.
 * Читает {@link ActionLogRecord} из входного итератора до тех пор, пока не будет
 * заполнен буфер, после чего делает запрос в таблицу ppclog_api для определения
 * application_id по reqid - это нужно, чтобы различать между собой различных пользователей
 * api директа.
 */
@ParametersAreNonnullByDefault
public class FillRecordStats implements PeekingIterator<ActionLogRecordWithStats> {
    private final PeekingIterator<ActionLogRecordWithStats> iterator;
    private final ReadRequestStats readRequestStats;
    private final ReadPpclogApiTable readPpclogApiTable;

    public FillRecordStats(
            int chunkSize, Iterator<ActionLogRecord> iterator, ReadPpclogApiTable readPpclogApiTable,
            ReadRequestStats readRequestStats) {
        if (chunkSize < 1) {
            throw new IllegalArgumentException("chunkSize should be positive");
        }

        this.iterator = Iterators.peekingIterator(
                Iterators.concat(Iterators.transform(Iterators.partition(iterator, chunkSize), this::addStats)));
        this.readPpclogApiTable = readPpclogApiTable;
        this.readRequestStats = readRequestStats;
    }

    private Iterator<ActionLogRecordWithStats> addStats(List<ActionLogRecord> records) {
        List<ActionLogRecord> apiRecords = new ArrayList<>();
        for (ActionLogRecord record : records) {
            Optional<String> service = record.getDirectTraceInfo().getService();
            if (service.isPresent() && API_SERVICES.contains(service.get())) {
                apiRecords.add(record);
            }
        }
        Map<Long, String> applicationIdByReqid = getApplicationIdByReqid(apiRecords);

        List<ActionLogRecordWithStats> result = new ArrayList<>();

        for (ActionLogRecord record : records) {
            OptionalLong reqid = record.getDirectTraceInfo().getReqId();
            String applicationId = reqid.isPresent() ? applicationIdByReqid.get(reqid.getAsLong()) : null;
            result.add(new ActionLogRecordWithStats(record, RecordStats.from(record, applicationId)));
        }
        readRequestStats.ppclogApiQueriesDone++;

        return result.iterator();
    }

    /**
     * Получить для списка {@link ActionLogRecord} application_id.
     *
     * @param records список записей, для которых нужно определить application_id
     * @return отображение reqid -> application_id
     */
    private Map<Long, String> getApplicationIdByReqid(List<ActionLogRecord> records) {
        SqlBuilder builder = readPpclogApiTable.sqlBuilder();
        readPpclogApiTable.applyTimeReqidFilter(builder, records);
        List<PpclogApiRecord> ppclogApiRecords = readPpclogApiTable.select(builder);
        Map<Long, String> result = new HashMap<>();
        for (PpclogApiRecord record : ppclogApiRecords) {
            result.put(record.getReqid(), record.getApplicationId());
        }
        return result;
    }

    @Override
    public ActionLogRecordWithStats peek() {
        return iterator.peek();
    }

    @Override
    public boolean hasNext() {
        return iterator.hasNext();
    }

    @Override
    public ActionLogRecordWithStats next() {
        return iterator.next();
    }

    @Override
    public void remove() {
        throw new UnsupportedOperationException();
    }
}
