package ru.yandex.direct.useractionlog.reader;

import java.time.LocalDate;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Objects;

import com.google.common.collect.Iterators;
import com.google.common.collect.PeekingIterator;

import ru.yandex.direct.useractionlog.schema.ActionLogRecord;
import ru.yandex.direct.useractionlog.schema.ObjectPath;

/**
 * Если итератор возвращает несколько записей с одинаковым первичным ключом, то выбирается запись с наибольшим
 * {@link ru.yandex.direct.useractionlog.schema.RecordSource}. Также выполняется фильтрация записей
 * с флагом {@link ActionLogRecord#isDeleted()}.
 */
class FilterByVersion implements PeekingIterator<ActionLogRecord> {
    private final PeekingIterator<ActionLogRecord> iterator;
    private PrimaryKey peekKey;
    private ActionLogRecord peekRecord;

    FilterByVersion(Iterator<ActionLogRecord> iterator) {
        this.iterator = Iterators.peekingIterator(iterator);
    }

    @Override
    public boolean hasNext() {
        skipWhileNeedAndWritePeek();
        return peekKey != null;
    }

    @Override
    public ActionLogRecord peek() {
        return peekRecord;
    }

    @Override
    public ActionLogRecord next() {
        if (!hasNext()) {
            throw new NoSuchElementException();
        }
        ActionLogRecord result = peekRecord;
        peekRecord = null;
        peekKey = null;
        return result;
    }

    @Override
    public void remove() {
        throw new UnsupportedOperationException();
    }

    private void skipWhileNeedAndWritePeek() {
        if (peekKey == null) {
            while (iterator.hasNext()) {
                peekRecordWithBiggestRecordSource();
                if (!peekRecord.isDeleted()) {
                    break;
                }
                peekKey = null;
                peekRecord = null;
            }
        }
    }

    private void peekRecordWithBiggestRecordSource() {
        peekRecord = iterator.next();
        peekKey = new PrimaryKey(peekRecord);
        while (iterator.hasNext()) {
            ActionLogRecord candidate = iterator.peek();
            if (peekKey.equals(new PrimaryKey(candidate))) {
                if (peekRecord.getRecordSource().compareTo(candidate.getRecordSource()) <= 0) {
                    peekRecord = candidate;
                }
                iterator.next();
            } else {
                break;
            }
        }
    }

    private static class PrimaryKey {
        private final LocalDate date;
        private final ObjectPath path;
        private final String gtid;
        private final int querySerial;
        private final int rowSerial;

        PrimaryKey(ActionLogRecord record) {
            date = record.getDateTime().toLocalDate();
            path = record.getPath();
            gtid = record.getGtid();
            querySerial = record.getQuerySerial();
            rowSerial = record.getRowSerial();
        }

        @Override
        public int hashCode() {
            return Objects.hash(date, path, gtid, querySerial, rowSerial);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            PrimaryKey that = (PrimaryKey) o;
            return querySerial == that.querySerial &&
                    rowSerial == that.rowSerial &&
                    Objects.equals(date, that.date) &&
                    Objects.equals(path, that.path) &&
                    Objects.equals(gtid, that.gtid);
        }

        @Override
        public String toString() {
            return "PrimaryKey{" +
                    "date=" + date +
                    ", path=" + path +
                    ", gtid='" + gtid + '\'' +
                    ", querySerial=" + querySerial +
                    ", rowSerial=" + rowSerial +
                    '}';
        }
    }
}
