package ru.yandex.direct.useractionlog.reader;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import com.google.common.collect.PeekingIterator;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.campaign.AvailableCampaignSources;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeSource;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.useractionlog.reader.generator.CampaignRestrictFrequencyEvent;
import ru.yandex.direct.useractionlog.reader.generator.FieldKeyValue;
import ru.yandex.direct.useractionlog.reader.generator.LogRecordGenerator;
import ru.yandex.direct.useractionlog.reader.model.BannersEvent;
import ru.yandex.direct.useractionlog.reader.model.CampaignEvent;
import ru.yandex.direct.useractionlog.reader.model.LogRecord;
import ru.yandex.direct.useractionlog.reader.model.OutputCategory;
import ru.yandex.direct.useractionlog.schema.ActionLogRecordWithStats;

import static ru.yandex.direct.common.db.PpcPropertyNames.FILTER_UAC_BANNERS_IN_ALW;

/**
 * Фильтрация событий на основании типа кампании, который может отсутствовать в записях из таблицы пользовательских
 * логов.
 * <p>
 * Реализуемые фильтрации:
 * <ul>
 * <li>DIRECT-79687 Кампании с типом wallet нельзя отображать в интерфейсе.</li>
 * <li>DIRECT-79685 События, связанные с ограничением показов, должны быть только в cpm и частных сделках.
 * В других кампаниях такой фичи нет.</li>
 * </ul>
 * У класса есть недостаток: он отправляет штучные запросы в ppc для получения типа кампании. Можно было бы отправлять
 * запросы в ppc пакетно, но так как заранее неизвестно необходимое количество записей, которые необходимо взять
 * из таблицы пользовательских логов, то выбор из двух зол: читать из таблицы логов больше, чем следует или отправлять
 * запросы в ppc поштучно.
 * <p>
 * По-хорошему синхронизатор должен сам отсеивать wallet-кампании и должен писать тип кампании в каждую строчку, чтобы
 * можно было фильтровать не обращаясь в ppc. Для этого придётся внести правку в синхронизатор (на момент написания
 * этого комментария это было несложной задачей) и добавить недостающее поле в уже записанные логи (а миграция данных
 * в кликхаусе - это сложная и кропотливая работа).
 * <p>
 * Если всё-таки запросов в ppc будет отправляться слишком много, а исправлять синхронизатор никто не захочет, то
 * можно сделать оптимизацию. На вход можно передавать запрошенный пользователем лимит сообщений и на первом проходе
 * собрать пачку событий с этим лимитом, а все последующие события также запрашивать поштучно.
 */
@Component
@ParametersAreNonnullByDefault
public class FilterLogRecordsByCampaignTypeBuilder {
    private final CampaignRepository campaignRepository;
    private final ShardHelper shardHelper;
    private final PpcProperty<Boolean> filterUacBanners;

    @Autowired
    public FilterLogRecordsByCampaignTypeBuilder(CampaignRepository campaignRepository,
                                                 ShardHelper shardHelper,
                                                 PpcPropertiesSupport ppcPropertiesSupport) {
        this.campaignRepository = campaignRepository;
        this.shardHelper = shardHelper;
        this.filterUacBanners = ppcPropertiesSupport.get(FILTER_UAC_BANNERS_IN_ALW, Duration.ofSeconds(60));
    }

    public FilterBuilder builder() {
        return new FilterBuilder();
    }

    public class FilterBuilder {
        private long clientId;

        public FilterBuilder withClientId(long clientId) {
            this.clientId = clientId;
            return this;
        }

        public Function<LogRecordGenerator, LogRecordGenerator> build() {
            Preconditions.checkState(clientId != 0, "forgotten clientId");
            return forward -> new FilterImpl(forward, clientId);
        }
    }

    private class FilterImpl implements LogRecordGenerator {
        private final LogRecordGenerator forward;
        private final Map<Long, Optional<CampaignTypeSource>> memoizedCampaignTypeSource;
        private final long clientId;

        FilterImpl(LogRecordGenerator forward, long clientId) {
            this.forward = forward;
            this.clientId = clientId;
            this.memoizedCampaignTypeSource = new HashMap<>();
        }

        @Override
        public List<LogRecord> offer(PeekingIterator<ActionLogRecordWithStats> recordIter) {
            List<LogRecord> forwardedResult = forward.offer(recordIter);
            List<LogRecord> result = new ArrayList<>(forwardedResult.size());
            filterOutEventsByCampaignType(forwardedResult, result);
            return result;
        }

        private void filterOutEventsByCampaignType(List<LogRecord> chunk, List<LogRecord> result) {
            // Несмотря на то, что в этом методе реализовано получение данных пачками, на самом деле
            // множество ниже будет состоять максимум из одного элемента. chunk генерируется из одной
            // строчки в таблице пользовательских логов, а у одной строчки есть только один ObjectPath.
            Set<Long> notCheckedCampaignIds = chunk.stream()
                    .map(LogRecord::getEvent)
                    .filter(e -> e instanceof CampaignEvent)
                    .map(CampaignEvent.class::cast)
                    .map(CampaignEvent::getCampaignId)
                    .filter(id -> !memoizedCampaignTypeSource.containsKey(id))
                    .collect(Collectors.toSet());
            if (!notCheckedCampaignIds.isEmpty()) {
                Map<Long, CampaignTypeSource> repositoryResponse = campaignRepository.getCampaignsTypeSourceMap(
                        shardHelper.getShardByClientId(ClientId.fromLong(clientId)),
                        notCheckedCampaignIds,
                        false);
                notCheckedCampaignIds.forEach(campaignId -> memoizedCampaignTypeSource.put(
                        campaignId,
                        Optional.ofNullable(repositoryResponse.get(campaignId))));
            }
            for (LogRecord logRecord : chunk) {
                boolean skip = false;
                if (logRecord.getEvent() instanceof CampaignEvent) {
                    CampaignEvent campaignEvent = (CampaignEvent) logRecord.getEvent();
                    var campaignTypeSource = memoizedCampaignTypeSource.get(campaignEvent.getCampaignId()).orElse(null);
                    var campaignType = campaignTypeSource != null ? campaignTypeSource.getCampaignType() : null;
                    if (campaignType != null) {
                        // DIRECT-79687, DIRECT-93693
                        skip = campaignType == CampaignType.WALLET || campaignType == CampaignType.BILLING_AGGREGATE;

                        // DIRECT-79685
                        skip |= logRecord.getEvent() instanceof CampaignRestrictFrequencyEvent &&
                                !EnumSet.of(
                                        CampaignType.CPM_BANNER,  // охватный продукт, оплата за показы, баннер в сети
                                        CampaignType.CPM_DEALS  // охватный продукт, показ по частным сделкам
                                ).contains(campaignType);
                    }
                }

                if (!skip && filterUacBanners.getOrDefault(false)
                        && logRecord.getEvent() instanceof BannersEvent) {
                    BannersEvent bannersEvent = (BannersEvent) logRecord.getEvent();
                    var campaignTypeSource = memoizedCampaignTypeSource.get(bannersEvent.getCampaignId()).orElse(null);

                    // Скрываем все операции с баннерами из мастеровых кампаний, т.к. баннеры пользователю не видны
                    // и по ним много дублирующих операций
                    skip = campaignTypeSource != null
                            && AvailableCampaignSources.INSTANCE.isUC(campaignTypeSource.getCampaignsSource());
                }

                if (!skip) {
                    result.add(logRecord);
                }
            }
        }

        @Override
        public Map<String, Collection<FieldKeyValue>> getSupportedActionLogRecordTypeToFields() {
            return forward.getSupportedActionLogRecordTypeToFields();
        }

        @Override
        public Map<String, Collection<FieldKeyValue>> getActionLogRecordTypeToFields(
                Collection<OutputCategory> categories) {
            return forward.getActionLogRecordTypeToFields(categories);
        }
    }
}
