package ru.yandex.direct.useractionlog.reader;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.function.BinaryOperator;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.apache.commons.lang3.tuple.ImmutablePair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.useractionlog.db.ReadActionLogTable;
import ru.yandex.direct.useractionlog.model.AutoChangeableSettings;
import ru.yandex.direct.useractionlog.model.AutoUpdatedSettingsEvent;
import ru.yandex.direct.useractionlog.model.RecommendationsManagementHistory;
import ru.yandex.direct.utils.CollectionUtils;
import ru.yandex.direct.utils.MonotonicTime;
import ru.yandex.direct.utils.NanoTimeClock;

import static ru.yandex.direct.useractionlog.db.ReadPpclogApiTable.USER_LOG_DATE_ZONE_ID;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

public class LastAutoUpdatedSettingsActionLogReader {

    private static final Logger logger = LoggerFactory.getLogger(LastAutoUpdatedSettingsActionLogReader.class);

    private final ReadActionLogTable readActionLogTable;

    public LastAutoUpdatedSettingsActionLogReader(ReadActionLogTable readActionLogTable) {
        this.readActionLogTable = readActionLogTable;
    }

    public Collection<AutoUpdatedSettingsEvent> getLastAutoUpdatedSettings(Long clientId,
                                                                           Collection<Long> targetObjectIds,
                                                                           Collection<AutoChangeableSettings> input,
                                                                           Integer maxTimeDepthInMonths){
        /**
         * Пока здесь поддерживаются только кампании, но в целом сделан задел для поддержки любых объектов (например, групп)
         * Для тех же групп достаточно сделать отдельный набор objectPaths,
         * в selectRecommendationManagementHistory по-прежнему передавать только пути кампаний (так как там ищем опции кампании)
         * а уже в selectLastAutoUpdatedSettings передать как кампании, так и группы
         */
        var campaignPaths = readActionLogTable.buildObjectPathsByCids(clientId, targetObjectIds);
        var maxTimeDepth = LocalDateTime.now().minusMonths(maxTimeDepthInMonths);
        MonotonicTime startTime = NanoTimeClock.now();
        var historyRecords =
                readActionLogTable.selectRecommendationManagementHistory(campaignPaths, maxTimeDepth, input);
        MonotonicTime endTime = NanoTimeClock.now();
        logger.info("Requested {} recommendation management history records from user_action_log table in {} seconds",
                historyRecords.size(),
                endTime.minus(startTime).toMillis() / 1e3);

        //оставляем только те path для которых выяснили что автоприменение включалось на горизонте maxTimeDepth
        var pathsWithEnabledRecManagement = historyRecords.stream()
                .filter(RecommendationsManagementHistory::isEnabled)
                .map(RecommendationsManagementHistory::getPath)
                .collect(Collectors.toList());

        if (CollectionUtils.isEmpty(pathsWithEnabledRecManagement)){
            //на всех кампаниях отключена опция автоприменения рекомендаций
            return Collections.emptyList();
        }

        //берем самую раннюю дату включения настройки среди всех кампаний чтобы сузить окно поиска в следующем запросе
        var minRecManagementEnableTime = historyRecords.stream()
                .filter(RecommendationsManagementHistory::isEnabled)
                .map(RecommendationsManagementHistory::getLastUpdateDateTime)
                .min(Comparator.naturalOrder())
                .orElseThrow()
                .atZone(USER_LOG_DATE_ZONE_ID)
                .toLocalDateTime();

        startTime = NanoTimeClock.now();
        var events = readActionLogTable.selectLastAutoUpdatedSettings(pathsWithEnabledRecManagement,
                minRecManagementEnableTime, input);
        endTime = NanoTimeClock.now();
        logger.info("Requested {} last autoupdate events from user_action_log table in {} seconds",
                events.size(),
                endTime.minus(startTime).toMillis() / 1e3);

        var result = enrichEventsWithSettingsData(input, events);
        var filteredResult = filterEventsBeforeRecommendationManagementEnabling(result, historyRecords);

        return enrichIsAutoEnabledData(historyRecords, filteredResult);
    }

    protected List<AutoUpdatedSettingsEvent> enrichEventsWithSettingsData(Collection<AutoChangeableSettings> settings,
                                                                        Collection<AutoUpdatedSettingsEvent> events){
        var settingsMap = listToMap(settings,
                s -> new ImmutablePair<>(s.getItem(), s.getSubitem()));

        return events.stream()
                .map(e -> e.withSettings(settingsMap.get(new ImmutablePair<>(e.getItem(), e.getSubitem())))
                        .withTargetObjectId(e.getPath().getId().toLong()))
                .collect(Collectors.toList());
    }

    /**
     * Для каждой записи об изменениях настроек добавляем также информацию о текущем состоянии галочки о возможности
     * автоматического применения рекомендаций
     */
    protected List<AutoUpdatedSettingsEvent> enrichIsAutoEnabledData(
            Collection<RecommendationsManagementHistory> historyRecords,
            List<AutoUpdatedSettingsEvent> events){
        var latestHistoryRecords = historyRecords.stream()
                .collect(Collectors.toMap(
                        r -> new ImmutablePair<>(r.getPath(), r.getRecommendationOptionName()), Function.identity(),
                        BinaryOperator.maxBy(Comparator.comparing(RecommendationsManagementHistory::getLastUpdateDateTime))));

        //Предполагаем для каждого события должна быть хотя бы одна historyRecord,
        // т.к. path без них мы отфильтровали перед вторым запросом
        return events.stream()
                .map(e -> e.withAutoApplyEnabled(latestHistoryRecords.get(
                        new ImmutablePair<>(e.getPath(), e.getSettings().getRecommendationOptionName())).isEnabled()))
                .collect(Collectors.toList());
    }

    /**
     * Для каждой кампании оставляем только те изменения которые случились после включения
     * соответствующей опции автоприменения рекомендаций
     */
    protected List<AutoUpdatedSettingsEvent> filterEventsBeforeRecommendationManagementEnabling(
                List<AutoUpdatedSettingsEvent> events, List<RecommendationsManagementHistory> historyRecords){
        var enabledHistoryRecords = historyRecords.stream()
                .filter(RecommendationsManagementHistory::isEnabled)
                .collect(Collectors.toList());

        var enabledHistoryRecordsMap = listToMap(enabledHistoryRecords,
                r -> new ImmutablePair<>(r.getPath(), r.getRecommendationOptionName()));

        return events.stream().filter(e -> isEventAfterHistoryRecord(e,
                        enabledHistoryRecordsMap.get(new ImmutablePair<>(e.getPath(),
                                e.getSettings().getRecommendationOptionName()))))
                .collect(Collectors.toList());
    }

    private boolean isEventAfterHistoryRecord(AutoUpdatedSettingsEvent event,
                                               RecommendationsManagementHistory historyRecord){
        if (historyRecord == null){
            return false;
        }

        return event.getLastAutoUpdateTime().isAfter(historyRecord.getLastUpdateDateTime());
    }

}
