package ru.yandex.direct.useractionlog.reader;

import java.time.LocalDateTime;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.function.Consumer;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.clickhouse.SqlBuilder;
import ru.yandex.direct.useractionlog.db.ReadActionLogTable;
import ru.yandex.direct.useractionlog.schema.ActionLogRecord;

/**
 * Чтение таблицы пользовательских логов по кускам.
 * Каждый раз весь результат сортируется в обратном порядке по дате, времени, gtid, querySerial, rowSerial - в точности,
 * как отдаётся пользователю. Считывается не более определённого количества записей за один раз. Если после первого
 * чтения в таблице остались ещё записи, то делается ещё один такой же запрос с таким же лимитом, но добавляется фильтр,
 * отсекающий все уже прочитанные записи.
 * <p>
 * Этот класс необходим для организации постраничного доступа, когда фильтром по дате, времени, gtid, querySerial и
 * rowSerial отсекаются записи, которые пользователь уже видел.
 * <p>
 * Также благодаря параметру limit удаётся разменять смерть от OutOfMemoryException на очень долгое получение
 * результата. В clickhouse на одну таблицу можно сделать только один индекс. У таблицы пользовательских логов индекс не
 * совпадает с применяемой сортировкой, поэтому обработка каждого запроса имеет линейно-логарифмическую трудоёмкость.
 * При чтении таблицы несколькими запросами требуется каждый раз сотрировать один и тот же объём данных, чтобы вернуть
 * из него срез.
 */
@ParametersAreNonnullByDefault
class OrderedChunkActionLogReader implements Iterator<ActionLogRecord> {
    private final ReadActionLogTable readActionLogTable;
    private final int limit;
    private final Consumer<SqlBuilder> applyFilter;
    private final ReadRequestStats readRequestStats;
    private final ReadActionLogTable.Order order;
    private Iterator<ActionLogRecord> recordIterator;
    @Nullable
    private ReadActionLogTable.Offset offset;
    private ReadActionLogTable.Offset previousOffset;
    private boolean readingFromLastChunk;

    /**
     * @param applyFilter      Обработчик, который накладывает необходимые фильтры на SqlBuilder.
     * @param limit            Максимальное количество записей, которое может быть прочитано за один раз и помещено в
     *                         оперативную память.
     * @param offset           Смещение на старте. Используется для постраничного доступа. Если null, то без смещения.
     * @param order            Сортировка по возрастанию или по убыванию даты и времени.
     * @param readRequestStats Объект, в который будет будет помещена статистика о проведённых SQL-запросах.
     */
    OrderedChunkActionLogReader(ReadActionLogTable readActionLogTable, Consumer<SqlBuilder> applyFilter,
                                int limit, @Nullable ReadActionLogTable.Offset offset, ReadActionLogTable.Order order,
                                ReadRequestStats readRequestStats) {
        this.readActionLogTable = readActionLogTable;
        this.limit = limit;
        this.applyFilter = applyFilter;
        this.readRequestStats = readRequestStats;
        this.order = order;
        this.recordIterator = Collections.emptyIterator();
        this.offset = offset;
        this.previousOffset = null;
    }

    /**
     * Возвращает фабрику, которую можно передать в {@link DateSplittingActionLogReader}.
     * Все аргументы будут переданы в
     * {@link #OrderedChunkActionLogReader(ReadActionLogTable, Consumer, int, ReadActionLogTable.Offset,
     * ReadActionLogTable.Order, ReadRequestStats)}.
     */
    static ActionLogRecordIteratorFactory factoryForFilter(ReadActionLogTable readActionLogTable, int limit,
                                                           ReadRequestStats readRequestStats) {
        return (applyFilter, offset, order) -> new OrderedChunkActionLogReader(
                readActionLogTable, applyFilter, limit, offset, order, readRequestStats);
    }

    @Override
    public boolean hasNext() {
        populateChunkIfNeed();
        return recordIterator.hasNext();
    }

    @Override
    public ActionLogRecord next() {
        populateChunkIfNeed();
        return recordIterator.next();
    }

    private void populateChunkIfNeed() {
        if (!readingFromLastChunk && !recordIterator.hasNext()) {
            SqlBuilder sqlBuilder = readActionLogTable.sqlBuilderWithSort(offset, order);
            applyFilter.accept(sqlBuilder);
            // Данные запрашиваются с нахлёстом в 1 запись, чтобы понять, есть ли ещё записи после лимита.
            // При подсчёте статистики эта запись не учитывается.
            sqlBuilder.limit(limit + 1);
            List<ActionLogRecord> result = readActionLogTable.select(sqlBuilder);
            ++readRequestStats.recordQueriesDone;
            readRequestStats.recordsFetched += result.size();
            if (result.size() >= limit + 1) {
                --readRequestStats.recordsFetched;
                offset = new SimpleOffset(result.remove(limit));
                if (previousOffset != null && previousOffset.equals(offset)) {
                    throw new RuntimeException("offset hasn't changed between two sql requests, limit is too low");
                }
                previousOffset = offset;
            } else {
                readingFromLastChunk = true;
                offset = null;
            }
            recordIterator = result.iterator();
        }
    }

    private static class SimpleOffset implements ReadActionLogTable.Offset {
        private final LocalDateTime dateTime;
        private final String gtid;
        private final int querySerial;
        private final int rowSerial;

        SimpleOffset(ActionLogRecord record) {
            dateTime = record.getDateTime();
            gtid = record.getGtid();
            querySerial = record.getQuerySerial();
            rowSerial = record.getRowSerial();
        }

        @Override
        public LocalDateTime getDateTime() {
            return dateTime;
        }

        @Override
        public String getGtid() {
            return gtid;
        }

        @Override
        public int getQuerySerial() {
            return querySerial;
        }

        @Override
        public int getRowSerial() {
            return rowSerial;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            SimpleOffset that = (SimpleOffset) o;
            return querySerial == that.querySerial &&
                    rowSerial == that.rowSerial &&
                    Objects.equals(dateTime, that.dateTime) &&
                    Objects.equals(gtid, that.gtid);
        }

        @Override
        public int hashCode() {
            return Objects.hash(dateTime, gtid, querySerial, rowSerial);
        }
    }
}
