package ru.yandex.direct.useractionlog.reader;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.function.BiFunction;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.PeekingIterator;

@ParametersAreNonnullByDefault
public class SplittingIteratorWindow<R, E, O extends SplittingIteratorWindow.Offset> {
    private final Function<PeekingIterator<R>, List<E>> mapper;
    private final BiFunction<R, Integer, O> offsetMaker;

    public SplittingIteratorWindow(Function<PeekingIterator<R>, List<E>> mapper,
                                   BiFunction<R, Integer, O> offsetMaker) {
        this.mapper = mapper;
        this.offsetMaker = offsetMaker;
    }

    public Result<E, O> process(PeekingIterator<R> iterator, int limit, @Nullable O offset) {
        if (limit <= 0) {
            throw new IllegalArgumentException("Non-positive limit");
        }
        int inRecordOffset = 0;
        if (offset != null) {
            inRecordOffset = offset.inRecordOffset();
        }
        List<E> result = new ArrayList<>(limit);
        R nextWindowFromRecord = null;
        int gotObjectsFromMapper = 0;
        while (result.size() < limit && iterator.hasNext()) {
            nextWindowFromRecord = iterator.peek();
            List<E> records = mapper.apply(iterator);
            gotObjectsFromMapper = records.size();
            for (int i = inRecordOffset; i < records.size(); ++i) {
                result.add(records.get(i));
            }
            if (inRecordOffset > records.size()) {
                inRecordOffset -= records.size();
            } else if (inRecordOffset > 0) {
                inRecordOffset = 0;
            }
        }
        O nextOffset = null;
        if (result.size() > limit) {
            int nextInRecordOffset = gotObjectsFromMapper - (result.size() - limit);
            if (nextInRecordOffset < 0) {
                throw new IllegalStateException("Bug. nextInRecordOffset == " + nextInRecordOffset);
            }
            nextOffset = offsetMaker.apply(
                    Objects.requireNonNull(nextWindowFromRecord),
                    nextInRecordOffset);
            result = result.subList(0, limit);
        } else if (result.size() == limit && iterator.hasNext()) {
            nextOffset = offsetMaker.apply(iterator.next(), 0);
        }
        return new Result<>(result, nextOffset);
    }

    public interface Offset {
        int inRecordOffset();
    }

    public static class Result<E, O> {
        private final List<E> objects;
        private final O nextPageOffset;

        public Result(List<E> objects, @Nullable O nextPageOffset) {
            this.objects = objects;
            this.nextPageOffset = nextPageOffset;
        }

        public List<E> getObjects() {
            return objects;
        }

        @Nullable
        public O getNextPageOffset() {
            return nextPageOffset;
        }
    }
}
